/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
/* samhainctl */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#ifdef HAVE_SETPRIORITY
#include <sys/resource.h>
#endif


#include "samhain.h"
#include "sh_files.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_unix.h"
#include "sh_getopt.h"
#include "sh_readconf.h"
#include "sh_hash.h"

#include "sh_mail.h"

#include "sh_tiger.h"
#include "sh_gpg.h"
#include "sh_mem.h"
#include "sh_forward.h"
#include "sh_tools.h"
#include "sh_hash.h"
#if defined(WITH_EXTERNAL)
#include "sh_extern.h"
#endif
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
#include "sh_modules.h"
#endif

#undef  FIL__
#define FIL__  _("samhain.c")


/**************************************************
 *
 * Needed to compile the key into the code.
 *
 **************************************************/

extern UINT32  ErrFlag[2];
#include "sh_MK.h"

/**************************************************
 *
 * Variables for signal handling.
 *
 **************************************************/

volatile  int      sig_raised;
volatile  int      sig_debug_on;           /* SIGUSR1 */
volatile  int      sig_debug_off;          /* SIGUSR2 */
volatile  int      sig_fresh_trail;        /* SIGIOT  */
volatile  int      sig_config_read_again;  /* SIGHUP  */
volatile  int      sig_terminate;          /* SIGQUIT */
long int           eintr__result;
char               sh_sig_msg[SH_MINIBUF];


#ifdef SH_STEALTH
/**************************************************
 *
 * The following set of functions is required for
 * the 'stealth' mode.
 *
 **************************************************/

#ifndef SH_MAX_GLOBS
#define SH_MAX_GLOBS 16
#endif

#ifndef GLOB_LEN
#define GLOB_LEN 255
#endif

char * globber(char * str)
{
  int i;
  int j;

  static   int  items = 0;
  static   int  count = 0;
  static   char glob[SH_MAX_GLOBS * (GLOB_LEN+1)];

  if (str == NULL)
    return NULL;
  else
    j = strlen(str);

  ++items;

  ASSERT((j <= GLOB_LEN), _("j <= GLOB_LEN"))

  if (j > GLOB_LEN) 
    j = GLOB_LEN;

  /* Overwrap the buffer.
   */
  if ( (count + j) >= (SH_MAX_GLOBS * (GLOB_LEN+1)))
    {
      count = 0;
      items = 0;
    }

  for (i = 0; i < j; ++i)
    {
      if (str[i] != '\n' && str[i] != '\t' && str[i] != '\r' && str[i] != '"')
	glob[count + i] = str[i] ^ XOR_CODE;
      else
	glob[count + i] = str[i];
    }
  glob[count + j] = '\0';

  i     = count;
  count = count + j + 1;
  return &glob[i];
}

void sh_do_encode (char * str, int len)
{
  register          int i;

  /* this is a symmetric operation
   */
  for (i = 0; i < len; ++i)
    {
      str[i] = str[i] ^ XOR_CODE;
    }
  return;
}

#endif

/**************************************************
 *
 * Global variables.
 *
 **************************************************/

sh_struct   sh;
sh_key_t  * skey = NULL;

extern unsigned char TcpFlag[8][PW_LEN+1];

/**************************************************
 *
 * Initializing.
 *
 **************************************************/

static int is_samhainctl_init = S_FALSE;

static
void sh_init (void)
{
  unsigned char * dez = NULL;
  int             i;
#if defined(SH_WITH_MAIL)
  char          * p;
  char            q[SH_PATHBUF];
#endif

  SL_ENTER(_("sh_init"));

#ifdef MKA_09
  ErrFlag[0] |= (1 << 8);
#endif
#ifdef MKA_10
  ErrFlag[0] |= (1 << 9);
#endif
#ifdef MKA_11
  ErrFlag[0] |= (1 << 10);
#endif
#ifdef MKA_12
  ErrFlag[0] |= (1 << 11);
#endif
#ifdef MKA_13
  ErrFlag[0] |= (1 << 12);
#endif
#ifdef MKA_14
  ErrFlag[0] |= (1 << 13);
#endif
#ifdef MKA_15
  ErrFlag[0] |= (1 << 14);
#endif
#ifdef MKA_16
  ErrFlag[0] |= (1 << 15);
#endif

  /* Signal handling.
   */
  sig_raised            = 0;
  sig_config_read_again = 0;           /* SIGHUP  */
  sig_debug_on          = 0;           /* SIGUSR1 */
  sig_debug_off         = 0;           /* SIGUSR2 */
  sig_fresh_trail       = 0;           /* SIGIOT  */
  sig_terminate         = 0;           /* SIGQUIT */
  strcpy ( sh_sig_msg, _("None"));

#ifdef MKB_01
  ErrFlag[1] |= (1 << 0);
#endif
#ifdef MKB_02
  ErrFlag[1] |= (1 << 1);
#endif
#ifdef MKB_03
  ErrFlag[1] |= (1 << 2);
#endif
#ifdef MKB_04
  ErrFlag[1] |= (1 << 3);
#endif
#ifdef MKB_05
  ErrFlag[1] |= (1 << 4);
#endif
#ifdef MKB_06
  ErrFlag[1] |= (1 << 5);
#endif
#ifdef MKB_07
  ErrFlag[1] |= (1 << 6);
#endif
#ifdef MKB_08
  ErrFlag[1] |= (1 << 7);
#endif

#if defined(SH_WITH_SERVER) && !defined(SH_WITH_CLIENT)
  strncpy(sh.prg_name, _("Yule"), 8);
  sh.prg_name[4] = '\0';
#else
  strncpy(sh.prg_name, _("Samhain"), 8);
  sh.prg_name[7] = '\0';
#endif

  /* The flags.
   */
  if (is_samhainctl_init == S_FALSE)
    sh.flag.checkSum        = SH_CHECK_NONE;
  sh.flag.update          = S_FALSE;
  sh.flag.opts            = S_FALSE;
  if (is_samhainctl_init == S_FALSE)
    sh.flag.isdaemon        = S_FALSE;
  sh.flag.isserver        = S_FALSE;
  sh.flag.islocked        = S_FALSE;
  sh.flag.smsg            = S_FALSE;
  sh.flag.log_start       = S_TRUE;
  sh.flag.reportonce      = S_TRUE;
  sh.flag.fulldetail      = S_FALSE;
  sh.flag.audit           = S_FALSE;
  sh.flag.nice            = 0;
  sh.flag.aud_mask        = 0xFFFFFFFFUL;
  sh.flag.client_severity = S_FALSE;
  sh.flag.client_class    = S_FALSE;
  sh.flag.hidefile        = S_FALSE;
  sh.flag.loop            = S_FALSE;

#ifdef MKB_09
  ErrFlag[1] |= (1 << 8);
#endif
#ifdef MKB_10
  ErrFlag[1] |= (1 << 9);
#endif
#ifdef MKB_11
  ErrFlag[1] |= (1 << 10);
#endif
#ifdef MKB_12
  ErrFlag[1] |= (1 << 11);
#endif
#ifdef MKB_13
  ErrFlag[1] |= (1 << 12);
#endif
#ifdef MKB_14
  ErrFlag[1] |= (1 << 13);
#endif
#ifdef MKB_15
  ErrFlag[1] |= (1 << 14);
#endif
#ifdef MKB_16
  ErrFlag[1] |= (1 << 15);
#endif

  /* The stats.
   */
  sh.stat.openfiles    = 0;
  sh.stat.bytes_hashed = 0;
  sh.stat.mail_success = 0;
  sh.stat.mail_failed  = 0;
  sh.stat.log_success  = 0;
  sh.stat.log_failed   = 0;
  sh.stat.time_start   = time(NULL);
  sh.stat.time_check   = (time_t) 0;

#ifdef MKC_01
  ErrFlag[0] |= (1 << 16);
#endif
#ifdef MKC_02
  ErrFlag[0] |= (1 << 17);
#endif
#ifdef MKC_03
  ErrFlag[0] |= (1 << 18);
#endif
#ifdef MKC_04
  ErrFlag[0] |= (1 << 19);
#endif
#ifdef MKC_05
  ErrFlag[0] |= (1 << 20);
#endif
#ifdef MKC_06
  ErrFlag[0] |= (1 << 21);
#endif
#ifdef MKC_07
  ErrFlag[0] |= (1 << 22);
#endif
#ifdef MKC_08
  ErrFlag[0] |= (1 << 23);
#endif


  /* The local host.
   */
  sl_strlcpy (sh.host.name,  _("localhost"),  SH_MINIBUF);
  sh.host.system[0]     = '\0';
  sh.host.release[0]    = '\0';
  sh.host.machine[0]    = '\0';

#ifdef MKC_09
  ErrFlag[0] |= (1 << 24);
#endif
#ifdef MKC_10
  ErrFlag[0] |= (1 << 25);
#endif
#ifdef MKC_11
  ErrFlag[0] |= (1 << 26);
#endif
#ifdef MKC_12
  ErrFlag[0] |= (1 << 27);
#endif
#ifdef MKC_13
  ErrFlag[0] |= (1 << 28);
#endif
#ifdef MKC_14
  ErrFlag[0] |= (1 << 29);
#endif
#ifdef MKC_15
  ErrFlag[0] |= (1 << 30);
#endif
#ifdef MKC_16
  ErrFlag[0] |= (1UL << 31);
#endif

  /* The paths.
   */
#if defined (SH_WITH_SERVER)
  sl_strlcpy (sh.conf.path,  DEFAULT_YULE_CONFIGFILE,    SH_PATHBUF);
#else
  sl_strlcpy (sh.conf.path,  DEFAULT_CONFIGFILE,    SH_PATHBUF);
#endif
  sh.conf.hash[0] = '\0';
  sl_strlcpy (sh.data.path,  DEFAULT_DATA_FILE,     SH_PATHBUF);
  sh.data.hash[0] = '\0';
  sh.exec.path[0] = '\0';
  sh.exec.hash[0] = '\0';

#ifdef MKD_01
  ErrFlag[1] |= (1 << 16);
#endif
#ifdef MKD_02
  ErrFlag[1] |= (1 << 17);
#endif
#ifdef MKD_03
  ErrFlag[1] |= (1 << 18);
#endif
#ifdef MKD_04
  ErrFlag[1] |= (1 << 19);
#endif
#ifdef MKD_05
  ErrFlag[1] |= (1 << 20);
#endif
#ifdef MKD_06
  ErrFlag[1] |= (1 << 21);
#endif
#ifdef MKD_07
  ErrFlag[1] |= (1 << 22);
#endif
#ifdef MKD_08
  ErrFlag[1] |= (1 << 23);
#endif

  /* The addresses.
   */
#if defined(SH_WITH_MAIL)
  if (0 == strcmp (DEFAULT_MAILADDRESS, _("NULL")))
    {
      sl_strncpy(q, DEFAULT_MAILADDRESS, SH_PATHBUF);
      p = strtok (q, ", \t");
      if (p)
	{
	  sh_mail_setaddress_int (p);
	  while (NULL != (p = strtok (NULL, ", \t")))
	    sh_mail_setaddress_int (p);
	}
    }
#endif

  if (0 == strcmp (ALT_TIMESERVER, _("NULL")))
    sh.srvtime.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvtime.alt, ALT_TIMESERVER,        SH_PATHBUF);
  if (0 == strcmp (DEFAULT_TIMESERVER, _("NULL")))
    sh.srvtime.name[0] = '\0';
  else
    sl_strlcpy (sh.srvtime.name, DEFAULT_TIMESERVER,   SH_PATHBUF);


  if (0 == strcmp (ALT_LOGSERVER, _("NULL")))
    sh.srvexport.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvexport.alt,  ALT_LOGSERVER,  SH_PATHBUF);
  if (0 == strcmp (DEFAULT_LOGSERVER, _("NULL")))
    sh.srvexport.name[0] = '\0';
  else
    sl_strlcpy (sh.srvexport.name,  DEFAULT_LOGSERVER, SH_PATHBUF);


#if defined (SH_WITH_SERVER)
  if (0 == strcmp (DEFAULT_YULE_ERRLOCK, _("NULL")))
    sh.srvlog.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvlog.alt,  DEFAULT_YULE_ERRLOCK,       SH_PATHBUF);
  if (0 == strcmp (DEFAULT_YULE_ERRFILE, _("NULL")))
    sh.srvlog.name[0] = '\0';
  else
    sl_strlcpy (sh.srvlog.name,  DEFAULT_YULE_ERRFILE,      SH_PATHBUF);
#else
  if (0 == strcmp (DEFAULT_ERRLOCK, _("NULL")))
    sh.srvlog.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvlog.alt,  DEFAULT_ERRLOCK,       SH_PATHBUF);
  if (0 == strcmp (DEFAULT_ERRFILE, _("NULL")))
    sh.srvlog.name[0] = '\0';
  else
    sl_strlcpy (sh.srvlog.name,  DEFAULT_ERRFILE,      SH_PATHBUF);
#endif

  if (0 == strcmp (ALT_CONSOLE, _("NULL")))
    sh.srvcons.alt[0] = '\0';
  else
    sl_strlcpy (sh.srvcons.alt,  ALT_CONSOLE,          SH_PATHBUF);
#ifndef DEFAULT_CONSOLE
  sl_strlcpy (sh.srvcons.name, _("/dev/console"),    SH_PATHBUF);
#else
  if (0 == strcmp (DEFAULT_CONSOLE, _("NULL")))
    sl_strlcpy (sh.srvcons.name, _("/dev/console"),    SH_PATHBUF);
  else
    sl_strlcpy (sh.srvcons.name,  DEFAULT_CONSOLE,     SH_PATHBUF);
#endif

#ifdef MKD_09
  ErrFlag[1] |= (1 << 24);
#endif
#ifdef MKD_10
  ErrFlag[1] |= (1 << 25);
#endif
#ifdef MKD_11
  ErrFlag[1] |= (1 << 26);
#endif
#ifdef MKD_12
  ErrFlag[1] |= (1 << 27);
#endif
#ifdef MKD_13
  ErrFlag[1] |= (1 << 28);
#endif
#ifdef MKD_14
  ErrFlag[1] |= (1 << 29);
#endif
#ifdef MKD_15
  ErrFlag[1] |= (1 << 30);
#endif
#ifdef MKD_16
  ErrFlag[1] |= (1UL << 31);
#endif


  /* The timers.
   */
  sh.fileCheck.alarm_last     = 0;
  sh.fileCheck.alarm_interval = 600; /* ten minutes */

  sh.mailTime.alarm_last     = 0;
  sh.mailTime.alarm_interval = 86400;

  sh.mailNum.alarm_last      = 0;
  sh.mailNum.alarm_interval  = 10;

  sh.looptime     = 60;


  /* The struct to hold privileged information.
   */
  skey = (sh_key_t *) sl_malloc (sizeof(sh_key_t));
  if (skey == NULL) 
    {
      perror(_("sh_init"));
      _exit (EXIT_FAILURE);
    }

  skey->mlock_failed = GOOD;
  skey->rngI         = BAD;

  skey->poolc        = 0;

  skey->ErrFlag[0]   = ErrFlag[0];
  ErrFlag[0]         = 0;
  skey->ErrFlag[1]   = ErrFlag[1];
  ErrFlag[1]         = 0;

  dez = &(TcpFlag[POS_TF-1][0]);
  for (i = 0; i < PW_LEN; ++i)
    { 
      skey->pw[i] = (*dez); 
      (*dez)      = '\0';
      ++dez; 
    }

  sh_unix_mlock();
  SL_RET0(_("sh_init"));
}


#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif

/*******************************************************
 * 
 * Exit Handler
 *
 *******************************************************/
void exit_handler(void)
{
  /* --- Clean up modules, if any. ---
   */
#if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE)
  int modnum;
#endif

  SL_ENTER(_("exit_handler"));

#if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE)
  for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
    {
      if (modList[modnum].initval == GOOD)
	modList[modnum].mod_cleanup();
    }
#endif

  /* --- Push out all pending messages. ---
   */
#if defined(SH_WITH_MAIL)
  if (sh.mailNum.alarm_last > 0) 
    {
      (void) sh_mail_msg (NULL);
    }
#endif

  /* --- Write the server stat. ---
   */
#if defined(SH_WITH_SERVER)
  sh_forward_html_write();
#endif

  /* --- Clean up memory to check for problems. ---
   */
#ifdef MEM_DEBUG
#if defined(SH_WITH_CLIENT) || defined(SH_STANDALONE)
  sh_files_deldirstack ();
  sh_files_delfilestack ();
  sh_hash_hashdelete();
#endif
#if defined(SH_WITH_SERVER)
  sh_forward_free_all ();
#endif
  delete_cache();
  sh_mem_stat();
#endif

  /* --- Checksum of executable. ---
   */
  sh_unix_self_check();


  /* --- Exit Message. ---
   */
  sh_error_handle ((-1), FIL__, __LINE__, sh.flag.exit, MSG_EXIT_NORMAL, 
		   sh.prg_name, sh_sig_msg);


  /* --- Restrict error logging to stderr. ---
   */
#ifdef WITH_MESSAGE_QUEUE
  close_ipc ();
#endif
  sh_error_only_stderr (S_TRUE);


  /* --- Remove lock, delete critical information. ---
   */
  sh_unix_unlock ();
  if (skey != NULL)
    memset (skey, '\0', sizeof(sh_key_t));
  
  /* --- Exit. ---
   */
  SL_RET0(_("exit_handler"));
}

/***********************************************************
 *
 */
#ifndef SIGHUP
#define SIGHUP   1
#endif
#ifndef SIGTERM
#define SIGTERM 15
#endif
#ifndef SIGKILL
#define SIGKILL  9
#endif

#if defined(__linux__) || defined(sun) || defined(__sun) || defined(__sun__)
#include <dirent.h>
static int * procdirSamhain ()
{
  int        * pidlist = malloc(sizeof(int) * 65535);
  struct dirent * d;
  DIR *        dp;
  long         ino;
  struct stat  buf;
  int          pid, i = 0;
  pid_t        mypid = getpid();
  char       * tail;
  char         exef[128];

  for (i = 0; i < 65535; ++i) pidlist[i] = 0;
  i = 0;

#ifdef SH_WITH_SERVER
  if (0 != lstat(SH_INSTALL_YULE_PATH, &buf))
    return NULL;
#else
  if (0 != lstat(SH_INSTALL_PATH, &buf))
    return NULL;
#endif
  ino = (long) buf.st_ino;
    
  if (NULL == (dp = opendir("/proc")))
    return NULL;
  while (NULL != (d = readdir(dp)) && i < 65535)
    {
      if (0 != strcmp(d->d_name, ".") && 0 != strcmp(d->d_name, ".."))
	{
	  errno = 0;
	  pid = strtol (d->d_name, &tail, 0);
	  if (*tail != '\0' || errno != 0)
	    continue;
	  if (pid == mypid)
	    continue;
#if defined(__linux__) 
	  sprintf(exef, _("/proc/%d/exe"), pid);         /* known to fit  */
#else
	  sprintf(exef, _("/proc/%d/object/a.out"), pid);/* known to fit  */
#endif
	  if (0 == stat(exef, &buf) && ino == (long) buf.st_ino)
	    { pidlist[i] = pid; ++i; }
	}
    }
  closedir(dp);
  return pidlist;
}
#else
static int * procdirSamhain ()
{
  return NULL;
}
#endif

static int killprocSamhain (pid_t pid)
{
  /* fprintf(stderr, "Killing %d\n", pid); */
  if (pid > 0 && 0 == kill (pid, SIGTERM))
    {
      sleep(1);
      if (0 != kill (pid, 0) && errno == ESRCH)
	return (0);
      sleep(2);
      if (0 != kill (pid, 0) && errno == ESRCH)
	return (0);
      sleep(2);
      if (0 != kill (pid, 0) && errno == ESRCH)
	return (0);
      
      kill (pid, SIGKILL);
      return (0);
    }
  
  return (1);
}

static int pidofSamhain (int flag)
{
  FILE      * fp;
  char        line[256];
  char      * tail;
  char      * p;
  pid_t       pid;
  struct stat buf;
 
#ifdef SH_WITH_SERVER
  fp = fopen (DEFAULT_YULE_ERRLOCK, "r");
#else
  fp = fopen (DEFAULT_ERRLOCK, "r");
#endif
  if (!fp)
    { if (errno != ENOENT) perror(_("fopen")); return 0; }
  if (NULL == fgets(line, 255, fp))
    { perror(_("fgets")); fclose(fp); return 0; }
  fclose(fp); 
  p = line; 
  while (*p == ' '  || *p == '\f' || *p == '\n' || 
	 *p == '\r' || *p == '\t' || *p == '\v')
    ++p;
  errno = 0;
  pid = strtol (p, &tail, 0);
  if (p == tail || errno != 0)
    { perror(_("strtol")); return 0; }

  /* remove stale pid file
   */
  if (flag == 1 && pid > 0 && 0 != kill(pid, 0) && errno == ESRCH)
    {
#ifdef SH_WITH_SERVER
      lstat (DEFAULT_YULE_ERRLOCK, &buf);
      if (S_ISREG(buf.st_mode)) unlink(DEFAULT_YULE_ERRLOCK);
#else
      lstat (DEFAULT_ERRLOCK, &buf);
      if (S_ISREG(buf.st_mode)) unlink(DEFAULT_ERRLOCK);
#endif
    }
  return pid;
}

/* 1: start 2:stop 3:reload 4:status
 */
static int samhainctl(int ctl, int * argc, char * argv[])
{
  char * fullpath;
  int    pid;
  int    status;
  int    res;
  int    times;
  char * argp[32];
  char * envp[2];
  int       * pidlist;
  int         i;


#ifdef SH_WITH_SERVER
  fullpath = malloc(strlen(SH_INSTALL_YULE_PATH)+1);
  if (fullpath == NULL)
    { perror(_("malloc")); exit (4); }
  else
    strcpy(fullpath, SH_INSTALL_YULE_PATH);            /* known to fit  */

  argp[0] = malloc(strlen(SH_INSTALL_YULE_PATH)+1);
  if (argp[0] == NULL)
    { perror(_("malloc")); exit (4); }
  else
    strcpy(argp[0], SH_INSTALL_YULE_PATH);             /* known to fit  */
#else
  fullpath = malloc(strlen(SH_INSTALL_PATH)+1);
  if (fullpath == NULL)
    { perror(_("malloc")); exit (4); }
  else
    strcpy(fullpath, SH_INSTALL_PATH);                 /* known to fit  */

  argp[0] = malloc(strlen(SH_INSTALL_PATH)+1);
  if (argp[0] == NULL)
    { perror(_("malloc")); exit (4); }
  else
    strcpy(argp[0], SH_INSTALL_PATH);                  /* known to fit  */
#endif

  for (times = 1; times < 32; ++times)  argp[times] = NULL;

  res = (*argc > 32) ? 32 : *argc;

  for (times = 2; times < res; ++times)  
    {
      argp[times-1] = malloc(strlen(argv[times])+1);
      if (argp[times-1] == NULL)
	{ perror(_("malloc")); exit (4); }
      else
	strcpy(argp[times-1], argv[times]);            /* known to fit  */
    }

  if (getenv("TZ") != NULL)
    {
      envp[0] = 
	malloc(strlen(getenv("TZ"))+1);
      if (envp[0] == NULL)
	{ perror(_("malloc")); exit (4); }
      else
	strcpy(envp[0], getenv("TZ"));                   /* known to fit  */
    }
  else
    {
      envp[0] = NULL;
    }
  envp[1] = NULL;

  if (ctl == 1)
    {
      pid = pidofSamhain(1);


      pid = fork();
      switch (pid) {
      case -1:
	perror(_("fork"));
	exit (4);
      case  0:
	close (0);
	execve(fullpath, argp, envp);
	perror(_("execve"));
	_exit (4);
      default:
	times = 0;
	while (times < 10) {
	  res = waitpid(pid, &status, WNOHANG|WUNTRACED);
	  if (-1 == res)
	    {
	      perror(_("waitpid"));
	      exit (4);
	    }
	  else if (pid == res)
	    {
	      if (0 != WIFEXITED(status))
		exit ( WEXITSTATUS(status) == 0 ? 0 : 1 );
	      else
		exit (1);
	    }
	  ++times;
	  sleep(1);
	}
	exit (1);
      }
    }

  pid = pidofSamhain(0);

  if (ctl == 2)  /* stop */
    {
      pidlist = procdirSamhain ();
      if (pid == 0 && NULL == pidlist)
	return (2);
	  
      status = 0;
      if (pid != 0)
	 status = killprocSamhain(pid);
      if (pidlist != NULL)
	{
	  i = 0; 
	  while (i < 65535 && pidlist[i] != 0)
	    { 
	      if (pidlist[i] != pid) 
		status = killprocSamhain(pidlist[i]);
	      ++i;
	    }
	}
      return status;
    }
	
  if (ctl == 3)  /* reload */
    {
      if (pid == 0)
	exit (2);
      if (0 == kill (pid, SIGHUP))
	exit (0);
      else
	exit (1);
    }

  if (ctl == 4)  /* status */
    {
      if (pid == 0)
	exit (2);
      if (0 == kill (pid, 0))
	exit (0);
      else
	exit (1);
    }

  exit (1); /* no exit handler installed yet */
}

/*******************************************************
 * 
 * Main program
 *
 *******************************************************/
int main(int argc, char * argv[])
{
#if defined(INET_SYSLOG)
  extern int    create_syslog_socket (int flag);
#endif

#if defined(SH_USE_XML)
  extern int    sh_log_file    (char * message);
#endif

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  int           modnum;
  time_t        runtim;
  float         st_1, st_2;
  int           status;
  long          cct = 0; /* main loop iterations */
#endif

  unsigned long told;
  unsigned long tcurrent;
  size_t        tzlen;
  int           res;

#if defined (SH_STEALTH_NOCL)
  char  * command_line;
  int     my_argc = 0;
  char  * my_argv[32];
#endif

  SL_ENTER(_("main"));

  if (argc >= 2 && 0 == getuid())
    {
      /* return codes:
       * 0    Success
       * 1    Can not send signal / start program
       * 2    Pid file does not exist
       */
      if      (0 == strcmp(argv[1], _("start")))
	{
	  samhainctl (1, &argc, argv);
	}
      else if (0 == strcmp(argv[1], _("stop")))
        return (samhainctl (2, &argc, argv));
      else if (0 == strcmp(argv[1], _("reload")))
	samhainctl (3, &argc, argv);
      else if (0 == strcmp(argv[1], _("status")))
	samhainctl (4, &argc, argv);
      else if (0 == strcmp(argv[1], _("restart")))
	{
	  res = samhainctl (2, &argc, argv);
	  if (res == 0 || res == 2)
	    {
	      samhainctl (1, &argc, argv);
	    }
	  else
	    return (res);
	}
    }
  
  if ((-1) == retry_fcntl(FIL__, __LINE__, 0, F_GETFL, 0) && 
	   errno == EBADF)
    {
      sh.flag.opts = S_TRUE;
      sh_unix_setdeamon(NULL);
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
      sh_util_setchecksum(_("check"));
#endif
      is_samhainctl_init = S_TRUE;
      sh.flag.opts = S_FALSE;
    }


  /* --- Install the exit handler. ---
   */
  (void) atexit(exit_handler);

  /* --- Zero the mailer key, and fill it. ---
   */
  memset (ErrFlag, 0, 2*sizeof(UINT32));

#ifdef MKA_01
  ErrFlag[0] |= (1 << 0);
#endif
#ifdef MKA_02
  ErrFlag[0] |= (1 << 1);
#endif
#ifdef MKA_03
  ErrFlag[0] |= (1 << 2);
#endif
#ifdef MKA_04
  ErrFlag[0] |= (1 << 3);
#endif
#ifdef MKA_05
  ErrFlag[0] |= (1 << 4);
#endif
#ifdef MKA_06
  ErrFlag[0] |= (1 << 5);
#endif
#ifdef MKA_07
  ErrFlag[0] |= (1 << 6);
#endif
#ifdef MKA_08
  ErrFlag[0] |= (1 << 7);
#endif

  BREAKEXIT(sh_derr);
  (void) sh_derr();

  /* Save the timezone.
   */
  if (getenv("TZ") != NULL)
    {
      tzlen       = strlen(getenv("TZ"));
      sh.timezone = sl_malloc (tzlen + 1);
      if (sh.timezone != NULL)
	sl_strlcpy (sh.timezone, getenv("TZ"), tzlen + 1);
    }
  else
    sh.timezone = NULL;
  


  /* --------  INIT  --------    
   */
#if defined(INET_SYSLOG) && defined(SH_WITH_SERVER)
  create_syslog_socket (S_TRUE);

  SL_REQUIRE(sl_policy_get_real(DEFAULT_IDENT) == SL_ENONE, 
	     _("sl_policy_get_real(DEFAULT_IDENT) == SL_ENONE"));
#else
  SL_REQUIRE(sl_policy_get_user(DEFAULT_IDENT) == SL_ENONE, 
	     _("sl_policy_get_user(DEFAULT_IDENT) == SL_ENONE"));
#endif

  /* Restrict error logging to stderr.
   */
  sh_error_only_stderr (S_TRUE);

  /* Check that first three descriptors are open.
   */
  if ( retry_fcntl(FIL__, __LINE__, 0, F_GETFL, 0) == (-1))
    aud_open(FIL__, __LINE__, SL_YESPRIV, _("/dev/null"), O_RDWR, 0);
  if ( retry_fcntl(FIL__, __LINE__, 1, F_GETFL, 0) == (-1))
    aud_open(FIL__, __LINE__, SL_YESPRIV, _("/dev/null"), O_RDWR, 1);
  if ( retry_fcntl(FIL__, __LINE__, 2, F_GETFL, 0) == (-1))
    aud_open(FIL__, __LINE__, SL_YESPRIV, _("/dev/null"), O_RDWR, 2);

  /* --- Set default values. ---
   */
  BREAKEXIT(sh_init);
  sh_init ();


#if (defined (SH_WITH_SERVER) && !defined (SH_WITH_CLIENT))
  sh.flag.isserver = S_TRUE;
#endif

  /* --- Get local hostname. ---
   */
  BREAKEXIT(sh_unix_localhost);
  sh_unix_localhost();

  /* --- Read the command line. ---
   */
  sh.flag.opts = S_TRUE;

#if !defined(SH_STEALTH_NOCL)
  (void) sh_getopt_get (argc, argv);
#else
  if (argc > 1 && argv[1] != NULL && 
      strlen(argv[1]) > 0 && strlen(NOCL_CODE) > 0)
    {
      if ( 0 == strcmp(argv[1], NOCL_CODE) &&
	   NULL != (command_line = (char *) SH_ALLOC(256 * sizeof(char))))
	{
	  my_argv[0] = argv[0]; ++my_argc;  
	  command_line[0] = '\0';
	  fgets (command_line, 255, stdin);
	  do {
	    my_argv[my_argc] = 
	      strtok( (my_argc == 1) ? command_line : NULL, " \n"); 
	    if (my_argv[my_argc] != NULL) {
	      ++my_argc;
	    } else {
	      break;
	    }
	  } while (my_argc < 32);
	  (void) sh_getopt_get (my_argc, my_argv);
	  SH_FREE (command_line);
	}
      else
	{
	  /* discard command line */
	  /* _exit(EXIT_FAILURE)  */  ; 
	}
    }
#endif
  sh.flag.opts = S_FALSE;
  

  /* --- Get user info. ---
   */
  TPT((0, FIL__, __LINE__, _("msg=<Get user hostname.>\n")))
  if (0 != sh_unix_getUser ())
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }


  /* *****************************
   *
   *  Read the configuration file.
   *
   * *****************************/

  TPT((0, FIL__, __LINE__, _("msg=<Read the configuration file.>\n")))
  BREAKEXIT(sh_readconf_read);
  (void) sh_readconf_read ();

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
  if (sh.flag.checkSum == SH_CHECK_NONE)
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
		       _("No action specified"), _("main"));
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit (FIL__, __LINE__, EXIT_FAILURE);
    }
#endif

#if defined(INET_SYSLOG) && defined(SH_WITH_SERVER)
  /* now check whether we really wanted it; if not, close
   */
  create_syslog_socket (S_FALSE);
#endif

  /* do not append to database if run SUID
   */
  if (sh.flag.checkSum == SH_CHECK_INIT && sl_is_suid()) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, EACCES, MSG_ACCESS,
		       (long) sh.real.uid, sh.data.path);
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }

  /* avoid daemon mode for initialization 
   */
  if (sh.flag.checkSum == SH_CHECK_INIT)
    sh.flag.isdaemon = S_FALSE;

  /* initialize signal handling etc
   */
  if (sh.flag.isdaemon == S_TRUE)
    { 
      sh_error_only_stderr (BAD);
#if defined(WITH_TRACE) || defined(WITH_TPT) 
      dbg_use_console();
#endif
    }

  if (sh_unix_init(sh.flag.isdaemon) == -1) 
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1,
		       sh.prg_name);
      aud_exit(FIL__, __LINE__, EXIT_FAILURE);
    }


  /* checksum of database
   */
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  TPT((0, FIL__, __LINE__, _("msg=<Get checksum of the database.>\n")))
  if (sh.flag.checkSum == SH_CHECK_CHECK) 
    {
      if (0 == sl_strcmp(file_path('D', 'R'), _("REQ_FROM_SERVER")))
	{
	  /* fetch the file from server to get checksum
	   */
	  sh_hash_init ();
	  sh_hash_hashdelete ();
	}
      else
	{
	  sl_strlcpy(sh.data.hash,
		     sh_tiger_hash (file_path('D', 'R'), TIGER_FILE, 0), 
		     KEY_LEN+1);
	}
    }
#endif


  /* --- Enable full error logging --- 
   */
  sh_error_only_stderr (S_FALSE);

  /****************************************************
   *
   *   SERVER 
   *
   ****************************************************/

#if defined(SH_WITH_SERVER) && !defined(SH_WITH_CLIENT)

#if (defined(WITH_GPG) || defined(WITH_PGP))
  /* do nothing -- we exit earlier if error 
  if (0 != sh_gpg_check_sign (1)) 
    aud_exit(FIL__, __LINE__, EXIT_FAILURE);
  */
#else
  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_1H,
		   sh.prg_name, (long) sh.real.uid, 
		   (sh.flag.hidefile == S_TRUE) ? 
		   _("(hidden)") : file_path('C','R'), 
		   sh.conf.hash);
#endif

#else

  /****************************************************
   *
   *   CLIENT/STANDALONE
   *
   ****************************************************/

  BREAKEXIT(sh_error_handle);

  if (sh.flag.checkSum == SH_CHECK_CHECK) 
    {
#if (defined(WITH_GPG) || defined(WITH_PGP))
      /* do nothing -- we exit earlier if error 
	 if (0 != sh_gpg_check_sign (2)) 
	 aud_exit(FIL__, __LINE__, EXIT_FAILURE);
      */
      ;
#else
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_2H,
		       sh.prg_name, (long) sh.real.uid,
		       (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('C', 'R'), sh.conf.hash,
		       (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('D', 'R'), sh.data.hash);
#endif
    }
  else
    {
#if (defined(WITH_GPG) || defined(WITH_PGP))
      /* do nothing -- we exit earlier if error 
      if (0 != sh_gpg_check_sign (1)) 
	aud_exit(FIL__, __LINE__, EXIT_FAILURE);
      */
      ;
#else
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_1H,
		       sh.prg_name, (long) sh.real.uid,
		       (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('C', 'R'), sh.conf.hash);
#endif
    }
#endif

 
  if (skey->mlock_failed == BAD)
    sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);

  /* timer
   */
  tcurrent                   = (unsigned long) time (NULL);
  told                       = tcurrent;
  sh.mailTime.alarm_last     = tcurrent;


  /****************************************************
   *
   *   SERVER 
   *
   ****************************************************/

#if defined(SH_WITH_SERVER)
  TPT((0, FIL__, __LINE__, _("msg=<Start server.>\n")))

#if defined (SH_WITH_CLIENT)
  if (sh.flag.isserver == S_TRUE)
    { 
      sh_receive();
      TPT((0, FIL__, __LINE__, _("msg=<End server.>\n")))
      aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
    }
#else
  sh_receive();
  TPT((0, FIL__, __LINE__, _("msg=<End server.>\n")))
  aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
#endif

#endif

  /****************************************************
   *
   *   CLIENT/STANDALONE
   *
   ****************************************************/
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)


  /* --- Initialize modules. ---
   */
  TPT((0, FIL__, __LINE__, _("msg=<Initialize modules.>\n")))
  for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
    {
      if (0 != (status = modList[modnum].mod_init()))
	{
	  sh_error_handle ((-1), FIL__, __LINE__, status, MSG_MOD_FAIL,
			   _(modList[modnum].name),
			   status);
	  modList[modnum].initval = S_FALSE;
	}
      else
	{
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_MOD_OK,
			   _(modList[modnum].name));
	  modList[modnum].initval = S_TRUE;
	}
    }
    
  /*  --------  TEST SETUP  ---------
   */
  sh_files_setrec();
  sh_files_test_setup();


  /* --------  NICE LEVEL   ---------
   */
  if (0 != sh.flag.nice)
    {
      sl_set_suid();
#ifdef HAVE_SETPRIORITY
      setpriority(PRIO_PROCESS, 0, sh.flag.nice);
#else
      nice(sh.flag.nice);
#endif
      sl_unset_suid();
    }

  /*  --------  MAIN LOOP  ---------
   */
  while (1) 
    {
      ++cct;

      BREAKEXIT(sh_error_handle);

      TPT((0, FIL__, __LINE__, _("msg=<Start main loop.>, iter=<%ld>\n"), cct))

      tcurrent                   = (unsigned long) time (NULL);

      if (sig_raised != 0)
	{

	  TPT((0, FIL__, __LINE__, _("msg=<Process a signal.>\n")))

	  if (sig_config_read_again == 1) /* SIGHUP */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Re-read configuration.>\n")))
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_RECONF);

#if defined(WITH_EXTERNAL)
	      /* delete list of external tasks
	       */
	      (void) sh_ext_cleanup();
#endif
	      /* delete the file list, make all database
	       * entries visible (allignore = FALSE)
	       */
	      sh_files_deldirstack ();
	      sh_files_delfilestack ();
	      hash_full_tree ();

#if defined(SH_WITH_CLIENT)
	      reset_count_dev_server();
#endif
#if defined(SH_WITH_MAIL)
	      reset_count_dev_mail();
#endif
	      reset_count_dev_console();
	      reset_count_dev_time();

	      sh_unix_maskreset();
 
	      /* Should this be included ??? 
	       * (i.e. should we reload the database ?)
	       */
#ifdef RELOAD_DATABASE
	      sh_hash_hashdelete();
#endif
	      sl_trust_add_user(-1);

	      (void) sh_readconf_read ();
	      sig_config_read_again = 0;
	      sh_files_setrec();
	      sh_files_test_setup();
	      if (0 != sh.flag.nice)
		{
		  sl_set_suid();
#ifdef HAVE_SETPRIORITY
		  setpriority(PRIO_PROCESS, 0, sh.flag.nice);
#else
		  nice(sh.flag.nice);
#endif
		  sl_unset_suid();
		}
	    }
	  
	  if (sig_fresh_trail == 1) /* SIGIOT */
	    {
	      /* Logfile access 
	       */
#ifdef SH_USE_XML
	      sh_log_file (NULL);
#endif
	      TPT((0, FIL__, __LINE__, _("msg=<Logfile stop/restart.>\n")))
	      sh_error_only_stderr (S_TRUE);
	      sh_unix_unlock();
	      sleep(3);
	      sh.flag.log_start = S_TRUE;
	      sh_error_only_stderr (S_FALSE);
	      sig_fresh_trail       = 0;
	    }
	  
	  if (sig_terminate == 1)  /* SIGQUIT */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Terminate.>\n")))
	      strncpy (sh_sig_msg, _("SIGQUIT"), 20);
	      aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
	    }
	  
	  if (sig_debug_on == 1)  /* SIGUSR1 */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Debug on.>\n")))
	      sh_error_dbg_on();
	      sig_debug_on = 0;
	    }
	  
	  if (sig_debug_off == 1)  /* SIGUSR2 */
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Debug off.>\n")))
	      sh_error_dbg_off();
	      sig_debug_off = 0;
	    }
	  sig_raised = 0;
	  TPT((0, FIL__, __LINE__, _("msg=<End signal processing.>\n")))
	}
      
      
      /* see whether its time to check files
       */
      if (sh.flag.checkSum != SH_CHECK_NONE &&
	  tcurrent - sh.fileCheck.alarm_last >= sh.fileCheck.alarm_interval) 
	{
	  /* 
	   * check directories and files
	   * ORDER IS IMPORTANT -- DIRZ FIRST
	   */
	  sh.stat.bytes_hashed  = 0;
	  sh.stat.time_start    = time (NULL);

	  TPT((0, FIL__, __LINE__, _("msg=<Check directories.>\n")))
	  BREAKEXIT(sh_dirs_chk);
	  sh.stat.dirs_checked  = sh_dirs_chk  (); 
	  TPT((0, FIL__, __LINE__, _("msg=<Check files.>\n")))
	  BREAKEXIT(sh_files_chk);
	  sh.stat.files_checked = sh_files_chk ();

	  if (sig_terminate == 1)
	    continue;

	  /*
	   * check for files not visited
	   */
	  TPT((0, FIL__, __LINE__, _("msg=<Check for missing files.>\n")))
	  sh_hash_unvisited (ShDFLevel[SH_ERR_T_FILE]);

	  if (sig_terminate == 1)
	    continue;

	  /* reset
	   */
	  TPT((0, FIL__, __LINE__, _("msg=<Reset status.>\n")))
	  sh_dirs_reset  ();
	  if (sig_terminate == 1)
	    continue;

	  sh_files_reset ();
	  if (sig_terminate == 1)
	    continue;

	  runtim = time(NULL) - sh.stat.time_start;

	
	  if (sh.stat.dirs_checked == 0 && sh.stat.files_checked == 0)
	    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_CHECK_0);

	  else
	    {
	      st_1 = sh.stat.bytes_hashed;
	      st_2 = runtim;


	      if (st_1 > 0.0 && st_2 > 0.0) 
		st_1 = st_1/st_2;
	      else if (st_1 > 0.0)
		st_1 = st_1 * 1.0;
	      else
		st_1 = 0.0;
	      st_1 = 0.001 * st_1;
	       
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_CHECK_1,
			       (long) runtim, 
			       st_1);
	    }
	  sh.fileCheck.alarm_last = (unsigned long) time (NULL);

	  if (sig_terminate == 1)
	    continue;

	  /*
	   * flush mail queue
	   */
#if defined(SH_WITH_MAIL)
	  TPT((0, FIL__, __LINE__, _("msg=<Flush mail queue.>\n")))
	  (void) sh_mail_msg (NULL);
#endif
	}
      
      if (sig_terminate == 1)
	continue;
      
      /* execute modules
       */
      TPT((0, FIL__, __LINE__, _("msg=<Execute modules.>\n")))
      for (modnum = 0; modList[modnum].name != NULL; ++modnum) 
	{
	  if (modList[modnum].initval == GOOD &&
	      0 != modList[modnum].mod_timer(tcurrent))
	    if (0 != (status = modList[modnum].mod_check()))
	      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_MOD_EXEC,
			       _(modList[modnum].name), (long) status);
	}
      
      /* no loop if not daemon
       */
      if (sh.flag.isdaemon != S_TRUE && sh.flag.loop == S_FALSE)
	break; 
      if (sig_terminate == 1)
	continue;
      
      /* see whether its time to send mail
       */
#if defined(SH_WITH_MAIL)
      if (tcurrent - sh.mailTime.alarm_last >= sh.mailTime.alarm_interval) 
	{
	  TPT((0, FIL__, __LINE__, _("msg=<Flush mail queue.>\n")))
	  (void) sh_mail_msg (NULL);
	  sh.mailTime.alarm_last = (unsigned long) time (NULL);
	}
#endif
      if (sig_terminate == 1)
	continue;
            
      /* log the timestamp
       */
      if ((tcurrent - told) >= (unsigned int) sh.looptime )
	{
	  TPT((0, FIL__, __LINE__, _("msg=<Log the timestamp.>\n")))
	  told = tcurrent;
#ifdef MEM_DEBUG
	  sh_mem_check();
#else
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_STAMP);
#endif
	}
    
      /* seed / re-seed the PRNG if required
       */
      (void) taus_seed();
      
      if (sig_terminate == 1)
	continue;
      
      /* go to sleep
       */
      (void) sleep (1);

      BREAKEXIT(sh_derr);
      (void) sh_derr();
    }
  
  /*   ------  END  -----------
   */

  /*
   * cleanup
   */
  TPT((0, FIL__, __LINE__, _("msg=<Cleanup.>\n")));
  sh_hash_hashdelete(); 

#if defined(SH_WITH_MAIL)
  if (sh.mailNum.alarm_last > 0) 
    (void)sh_mail_msg (NULL);
#endif

  /* #if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) */
#endif

  aud_exit (FIL__, __LINE__, EXIT_SUCCESS);
  SL_RETURN(0, _("main"));
}
