/***************************************************************************
 *
 * Purpose:
 * -------
 *   (1) Hide files with the string MAGIC_HIDE in filename,
 *       where MAGIC_HIDE is defined below. 
 *       By default,  MAGIC_HIDE is defined as "samhain".
 *
 *   (2) Hide all processes, if the executable has the string MAGIC_HIDE 
 *       in its name.
 *
 *
 * Configuration:
 * -------------
 *   If not building within the samhain system, you may remove the 
 *   line '#include "config.h"' and in the line
 *   '#define MAGIC_HIDE SH_MAGIC_HIDE', replace SH_MAGIC_HIDE with
 *   "someString" (in quotes !).
 */

#include "config.h" 
#define MAGIC_HIDE SH_MAGIC_HIDE

/*  #define MAGIC_HIDE "someString"              */

/* define this if you have a modversioned kernel */
/*  #define MODVERSIONS                           */

/*
 * Install:
 * -------
 *   gcc -Wall -O2 -c samhain_hide.c
 *   mv samhain_hide.o  /lib/modules/KERNEL_VERSION/misc/
 *   
 *   (Replace KERNEL_VERSION with your kernel's version.)
 *
 * Usage:
 * -----
 *   To load the module:
 *    insmod samhain_hide (for improved safety: 'sync && insmod samhain_hide')
 *
 *   To unload the module 
 *    rmmod samhain_hide  (for improved safety: 'sync && rmmod samhain_hide')
 * 
 *
 * Details:
 * -------
 *   The following kernel syscalls are replaced:
 *     sys_getdents     [hide files/directories/processes (/proc/PID)]
 * 
 * Tested on:
 * ---------
 *   Linux 2.2
 *
 * Copyright:
 * ---------
 *   Copyright (C) 2001 Rainer Wichmann (http://la-samhna.de)
 *
 * License: 
 * -------
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License, version 2, as
 *   published by the Free Software Foundation.
 *                                                                         
 *   This program is distributed in the hope that it will be useful,        
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of         
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          
 *   GNU General Public License for more details.                           
 *                                                                         
 *   You should have received a copy of the GNU General Public License      
 *   along with this program; if not, write to the Free Software            
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
 *
 ***************************************************************************/



/*****************************************************
 *
 *  The defines:
 *
 *****************************************************/

/* This is a Linux Loadable Kernel Module.
 */

#define __KERNEL__
#define MODULE
#define LINUX

/* Define for debugging.   
 */
/* #define HIDE_DEBUG  */   /* query_module */
/* #define FILE_DEBUG  */   /* getdents     */
/* #define READ_DEBUG  */   /* read         */

/*****************************************************
 *
 *  The include files:
 *
 *****************************************************/


/* The configure options (#defines) for the Kernel
 */
#include <linux/config.h>

#ifdef CONFIG_MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>

/* File tables structures. If directory caching is used,
 * <linux/dcache.h> will be included here, and __LINUX_DCACHE_H
 * will thus be defined.
 */
#include <linux/fs.h>
#include <linux/proc_fs.h>

/* Include the SYS_syscall defines.
 */
#include <sys/syscall.h>

/* Includes for 'getdents' per the manpage.
 */
#include <linux/types.h>
#include <linux/dirent.h>
#include <linux/unistd.h>

/* To access userspace memory.
 */
#include <asm/uaccess.h>

/* Include for lock_kernel().
 */
#include <linux/smp_lock.h>

/* Include for fget().
 */
#include <linux/file.h>

/*****************************************************
 *
 *  The global variables:
 *
 *****************************************************/

/* The kernel syscall table.
 */
extern void *sys_call_table[];

/* The old address of the sys_getdents syscall.
 */
int (*old_getdents)(unsigned int, struct dirent *, unsigned int);
#ifdef __NR_getdents64
long (*old_getdents64)(unsigned int, struct dirent64 *, unsigned int);
#endif

char hidden[] = MAGIC_HIDE;
 

/*****************************************************
 *
 *  The functions:
 *
 *****************************************************/


MODULE_AUTHOR("Rainer Wichmann");
MODULE_DESCRIPTION("Hide files/processes/modules with MAGIC_HIDE in name.");



/* Fetch the task struct for a given PID.
 */
struct task_struct * fetch_task_struct (int pid)
{
  struct task_struct * task_ptr;

  task_ptr = current;

  do 
    {
      if (task_ptr->pid == (pid_t) pid )
	return (task_ptr);
      task_ptr = task_ptr->next_task;
    } 
  while (task_ptr != current);

  return (NULL);
}


/* Convert a string to an int. 
 * Does not recognize integers with a sign (+/-) in front.
 */
int my_atoi(char * in_str)
{
  int i      = 0;
  int retval = 0;
  int conv   = 0;

  if (in_str == NULL)
    return (-1);

  while(in_str[i] != '\0')
    {
      /* Break if not numeric.
       */
      if (in_str[i] < '0' || in_str[i] > '9')
	break;

      ++conv;
      
      /* Leading zeroes (should not happen in /proc)
       */
      if (retval == 0 && in_str[i] == '0')
	retval = retval;
      else
	retval = retval * 10;

      retval = retval + (in_str[i] - '0');

      i++;
    }
      
  if (conv == 0)
    return (-1);
  else
    return (retval); 
}

/* Purpose:
 * 
 *   Hide all files/dirs that include the string MAGIC_HIDE in their
 *   name. 
 */
int new_getdents (unsigned int fd, struct dirent *dirp, unsigned int count)
{
  int                  status = 0;    /* Return value from original getdents */
  struct inode       * dir_inode;
  int                  dir_is_proc = 0;
  struct dirent      * dirp_prev;
  struct dirent      * dirp_new;
  struct dirent      * dirp_current;
  int                  dir_table_bytes;
  int                  forward_bytes;
  struct task_struct * task_ptr;
  int                  hide_it = 0;
  long                 dirp_offset;

  lock_kernel();

  status = (*old_getdents)(fd, dirp, count);

#ifdef FILE_DEBUG
  printk("STATUS %d\n", status);
#endif

  /*  0: end of directory.
   * -1: some error
   */
  if (status <= 0)
    {
      unlock_kernel();
      return (status);
    }

  /* Handle directory caching. dir_inode is the inode of the directory.
   */
#if defined(__LINUX_DCACHE_H)
  dir_inode  = current->files->fd[fd]->f_dentry->d_inode;
#else
  dir_inode  = current->files->fd[fd]->f_inode;
#endif

  /* Check for the /proc directory
   */
  if (dir_inode->i_ino == PROC_ROOT_INO && 
      !MAJOR(dir_inode->i_dev) && 
      MINOR(dir_inode->i_dev) == 1)
    dir_is_proc = 1;

  /* Allocate space for new dirent table. Can't use GFP_KERNEL 
   * (kernel oops)
   */
  dirp_new = (struct dirent *) kmalloc (status, GFP_ATOMIC);
  if (dirp_new == NULL)
    {
      unlock_kernel();
      return (status);
    }

  /* Copy the dirp table to kernel space.
   */
  copy_from_user(dirp_new, dirp, status);

#ifdef FILE_DEBUG
  printk("COPY to kernel\n");
#endif

  /* Loop over the dirp table to find entries to hide.
   */
  dir_table_bytes = status;
  dirp_current    = dirp_new;
  dirp_prev       = NULL;

  while (dir_table_bytes > 0)
    {
      hide_it = 0;

      if (dirp_current->d_reclen == 0)
	break;

      dirp_offset = dirp_current->d_off;
      
#ifdef FILE_DEBUG
      printk("DIRENT %d  %d  %ld\n", 
	     dir_table_bytes,
	     dirp_current->d_reclen,
	     dirp_current->d_off);
#endif

      dir_table_bytes -= dirp_current->d_reclen;
      forward_bytes    = dirp_current->d_reclen;

#ifdef FILE_DEBUG
      printk("ENTRY %s\n", dirp_current->d_name);
#endif

      /* If /proc is scanned (e.g. by 'ps'), hide the entry for
       * any process where the executable has MAGIC_HIDE in its name.
       */
      if (dir_is_proc == 1)
	{
	  task_ptr = fetch_task_struct(my_atoi(dirp_current->d_name));
	  if (task_ptr != NULL)
	    {
	      if (strstr(task_ptr->comm, hidden) != NULL)
		hide_it = 1;
	    }
	}
      /* If it is a regular directory, hide any entry with
       * MAGIC_HIDE in its name.
       */
      else
	{
	  if (strstr (dirp_current->d_name, hidden) != NULL)
	    hide_it = 1;
	}

      if (hide_it == 1)
	{
#ifdef FILE_DEBUG
	  printk("  -->HIDDEN %s\n", dirp_current->d_name);
#endif
	  if (dir_table_bytes > 0)
	    {
	      status -= dirp_current->d_reclen;
	      memmove (dirp_current, 
		       (char *) dirp_current + dirp_current->d_reclen, 
		       dir_table_bytes);

	      /* Set forward_bytes to 0, because now dirp_current is the
	       * (previously) next entry in the dirp table.
	       */
	      forward_bytes    = 0;
	      dirp_prev        = dirp_current;
	    }
	  else
	    {
	      status -= dirp_current->d_reclen;
	      if (dirp_prev != NULL)
		dirp_prev->d_off = dirp_offset;
	    }
	  
	}
      else
	{
	  dirp_prev        = dirp_current;
	  if (dir_table_bytes == 0 && dirp_prev != NULL)
	    dirp_prev->d_off = dirp_offset;
	}

      /* Next entry in dirp table.
       */
      if (dir_table_bytes > 0)
	dirp_current = (struct dirent *) ( (char *) dirp_current + 
					   forward_bytes);
    }

  /* Copy our modified dirp table back to user space.
   */
  copy_to_user(dirp, dirp_new, status);
#ifdef FILE_DEBUG
  printk("COPY to user\n");
#endif

  kfree (dirp_new);
#ifdef FILE_DEBUG
  printk("KFREE\n");
#endif

  unlock_kernel();
  return (status);
}

/* For 2.4 kernel
 */
#ifdef __NR_getdents64
long new_getdents64 (unsigned int fd, struct dirent64 *dirp, 
		     unsigned int count)
{
  long                 status = 0;    /* Return value from original getdents */
  struct inode       * dir_inode;
  int                  dir_is_proc = 0;
  struct dirent64    * dirp_prev;
  struct dirent64    * dirp_new;
  struct dirent64    * dirp_current;
  int                  dir_table_bytes;
  int                  forward_bytes;
  struct task_struct * task_ptr;
  int                  hide_it = 0;
  __s64                dirp_offset;

  lock_kernel();

  status = (*old_getdents64)(fd, dirp, count);

#ifdef FILE_DEBUG
  printk("STATUS64 %ld\n", status);
#endif

  /*  0: end of directory.
   * -1: some error
   */
  if (status <= 0)
    {
      unlock_kernel();
      return (status);
    }

  /* Handle directory caching. dir_inode is the inode of the directory.
   */
#if defined(__LINUX_DCACHE_H)
  dir_inode  = current->files->fd[fd]->f_dentry->d_inode;
#else
  dir_inode  = current->files->fd[fd]->f_inode;
#endif

#ifdef FILE_DEBUG
  printk("INODE64\n");
#endif

  /* Check for the /proc directory
   */
  if (dir_inode->i_ino == PROC_ROOT_INO && 
      !MAJOR(dir_inode->i_dev) && 
      MINOR(dir_inode->i_dev) == 1)
    dir_is_proc = 1;

#ifdef FILE_DEBUG
  printk("PROC_CHECK64\n");
#endif

  /* Allocate space for new dirent table. Can't use GFP_KERNEL 
   * (kernel oops)
   */
  dirp_new = kmalloc ((size_t)status, GFP_ATOMIC);

#ifdef FILE_DEBUG
  printk("KMALLOC64_0\n");
#endif

  if (dirp_new == NULL)
    {
      unlock_kernel();
      return (status);
    }

#ifdef FILE_DEBUG
  printk("KMALLOC64\n");
#endif

  /* Copy the dirp table to kernel space.
   */
  copy_from_user(dirp_new, dirp, status);

#ifdef FILE_DEBUG
  printk("COPY64 to kernel\n");
#endif

  /* Loop over the dirp table to find entries to hide.
   */
  dir_table_bytes = status;
  dirp_current    = dirp_new;
  dirp_prev       = NULL;

  while (dir_table_bytes > 0)
    {
      hide_it = 0;

      if (dirp_current->d_reclen == 0)
	break;

      dirp_offset = dirp_current->d_off;
      
#ifdef FILE_DEBUG
      printk("DIRENT %d  %d  %lld\n", 
	     dir_table_bytes,
	     dirp_current->d_reclen,
	     dirp_current->d_off);
#endif

      dir_table_bytes -= dirp_current->d_reclen;
      forward_bytes    = dirp_current->d_reclen;

#ifdef FILE_DEBUG
      printk("ENTRY %s\n", dirp_current->d_name);
#endif

      /* If /proc is scanned (e.g. by 'ps'), hide the entry for
       * any process where the executable has MAGIC_HIDE in its name.
       */
      if (dir_is_proc == 1)
	{
	  task_ptr = fetch_task_struct(my_atoi(dirp_current->d_name));
	  if (task_ptr != NULL)
	    {
	      if (strstr(task_ptr->comm, hidden) != NULL)
		hide_it = 1;
	    }
	}
      /* If it is a regular directory, hide any entry with
       * MAGIC_HIDE in its name.
       */
      else
	{
	  if (strstr (dirp_current->d_name, hidden) != NULL)
	    hide_it = 1;
	}

      if (hide_it == 1)
	{
#ifdef FILE_DEBUG
	  printk("  -->HIDDEN %s\n", dirp_current->d_name);
#endif
	  if (dir_table_bytes > 0)
	    {
	      status -= dirp_current->d_reclen;
	      memmove (dirp_current, 
		       (char *) dirp_current + dirp_current->d_reclen, 
		       dir_table_bytes);

	      /* Set forward_bytes to 0, because now dirp_current is the
	       * (previously) next entry in the dirp table.
	       */
	      forward_bytes    = 0;
	      dirp_prev        = dirp_current;
	    }
	  else
	    {
	      status -= dirp_current->d_reclen;
	      if (dirp_prev != NULL)
		dirp_prev->d_off = dirp_offset;
	    }
	  
	}
      else
	{
	  dirp_prev        = dirp_current;
	  if (dir_table_bytes == 0 && dirp_prev != NULL)
	    dirp_prev->d_off = dirp_offset;
	}

      /* Next entry in dirp table.
       */
      if (dir_table_bytes > 0)
	dirp_current = (struct dirent64 *) ( (char *) dirp_current + 
					     forward_bytes);
    }

  /* Copy our modified dirp table back to user space.
   */
  copy_to_user(dirp, dirp_new, status);
  kfree (dirp_new);
  unlock_kernel();
  return (status);
}
#endif

/* The initialisation function. Automatically called when module is inserted
 * via the 'insmod' command.
 */
int init_module(void)
{

  lock_kernel();

  /* Unfortunately this does not fully prevent the module from appearing
   * in /proc/ksyms. 
   */

  EXPORT_NO_SYMBOLS;

  /* Replace the 'sys_getdents' syscall with the new version.
   */
  old_getdents                     = sys_call_table[SYS_getdents];
  sys_call_table[SYS_getdents]     = new_getdents;
  
#ifdef __NR_getdents64
  old_getdents64                   = sys_call_table[SYS_getdents64];
  sys_call_table[SYS_getdents64]   = new_getdents64;
#endif

  unlock_kernel();
  return (0);
}

/* The cleanup function. Automatically called when module is removed
 * via the 'rmmod' command.
 */
void cleanup_module(void)
{
  lock_kernel();

  /* Restore the new syscalls to the original version.
   */
  sys_call_table[SYS_getdents]     = old_getdents;
#ifdef __NR_getdents64
  sys_call_table[SYS_getdents64]   = old_getdents64;
#endif

  unlock_kernel();
}




