/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <stdlib.h>


#if defined(WITH_GPG) || defined(WITH_PGP)

#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/wait.h>

#include <string.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif


#if !defined(O_NONBLOCK)
#if defined(O_NDELAY)
#define O_NONBLOCK  O_NDELAY
#else
#define O_NONBLOCK  0
#endif
#endif


#include "samhain.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_tiger.h"

static struct {
  char     conf_id[SH_MINIBUF+1];
  char     conf_fp[SH_MINIBUF+1];
  char     data_id[SH_MINIBUF+1];
  char     data_fp[SH_MINIBUF+1];
} gp;

typedef struct {
  pid_t    pid;
  FILE   * pipe;
} sh_gpg_popen_t;

#define SH_GPG_OK      0
#define SH_GPG_BAD     1
#define SH_GPG_BADSIGN 2

#undef  FIL__
#define FIL__  _("sh_gpg.c")

#ifdef GPG_HASH
static int sh_gpg_checksum (SL_TICKET checkfd, int flag)
{
  char * test_gpg;
  char * test_ptr1 = NULL;
  char * test_ptr2 = NULL;

  SL_ENTER(_("sh_gpg_checksum"));

  tiger_fd = checkfd;
#if defined(WITH_PGP)
  test_gpg = sh_tiger_hash_gpg (DEFAULT_PGP_PATH, TIGER_FD, 0);
#else
  test_gpg = sh_tiger_hash_gpg (DEFAULT_GPG_PATH, TIGER_FD, 0);
#endif
  
  test_ptr1 = strchr(GPG_HASH, ':');
  if (test_gpg != NULL)
    test_ptr2 = strchr(test_gpg, ':');
  
  if (test_ptr2 != NULL)
    test_ptr2 += 2;
  else
    test_ptr2 = test_gpg;
  if (test_ptr1 != NULL)
    test_ptr1 += 2;
  else
    test_ptr1 = GPG_HASH;
  
  if (0 != sl_strncmp(test_ptr1, test_ptr2, KEY_LEN + 6))
    {
      TPT(((0), FIL__, __LINE__, _("msg=<pgp checksum: %s>\n"), test_gpg));
      TPT(((0), FIL__, __LINE__, _("msg=<Compiled-in : %s>\n"), GPG_HASH));
      if (flag == 1)
	sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_GPG, 
			GPG_HASH, test_gpg);
      SH_FREE(test_gpg);
      SL_RETURN((-1), _("sh_gpg_checksum"));
    }
  SH_FREE(test_gpg);
  SL_RETURN( (0), _("sh_gpg_checksum"));
}
#endif

static FILE * sh_gpg_popen (sh_gpg_popen_t  *source, int fd, 
			    int mode, char * id)
{
  int pipedes[2];
  FILE * outf = NULL;
  char * envp[2];
  char   path[256];
  char   cc1[32];
  char   cc2[32];
#if defined(WITH_PGP)
  char   cc3[32];
  char   cc0[3] = "-f";
#endif
#if defined(WITH_GPG)
  char   cc0[2] = "-";
#endif

  char * arg[6];

#if defined(HAVE_GPG_CHECKSUM)
  SL_TICKET   checkfd;
  int         myrand;
  int         i;
#if defined(__linux__)
  int         get_the_fd(SL_TICKET);
  char        pname[128];
  int         pfd;
#endif
#endif

  SL_ENTER(_("sh_gpg_popen"));

#if defined(WITH_GPG)
  /* -- GnuPG -- */
  sl_strlcpy (path,  DEFAULT_GPG_PATH, 256);
  sl_strlcpy (cc1,   _("--status-fd"),  32);
  sl_strlcpy (cc2,   _("--verify"),     32);


  arg[0] = path; 
  arg[1] = cc1;
  arg[2] = "1";
  arg[3] = cc2;
  arg[4] = cc0;
  arg[5] = NULL;

  /* catch 'unused parameter' compiler warning
   */
  (void) mode;
  (void) id;
#elif defined(WITH_PGP)
  /* -- PGP -- */
  sl_strlcpy (path,  DEFAULT_PGP_PATH, 256);
  if (mode == 0)
    {
      sl_strlcpy (cc1,   _("+language=en"),  32);
      sl_strlcpy (cc2,   _("-o"),     32);
      sl_strlcpy (cc3,   _("/dev/null"),     32);
      
      arg[0] = path; 
      arg[1] = cc1;
      arg[2] = cc2; 
      arg[3] = cc3; 
      arg[4] = cc0;
      arg[5] = NULL;
    }
  else
    {
      sl_strlcpy (cc1,   _("+language=en"),  32);
      sl_strlcpy (cc2,   _("-kvc"),     32);       
      
      arg[0] = path; 
      arg[1] = cc1;
      arg[2] = cc2;
      arg[3] = id;
      arg[4] = NULL;
      arg[5] = NULL;
    }
#endif

  /* use homedir of effective user
   */
  if (sh.effective.home != NULL)
    {
      envp[0] = sl_malloc (sl_strlen(sh.effective.home) + 6);
      if (envp[0] != NULL)
	sprintf (envp[0], "HOME=%s",                     /* known to fit  */
		 sh.effective.home); 
      envp[1] = NULL;
    }
  else
    {
      envp[0] = NULL;
    }

  /* Create the pipe 
   */
  if (aud_pipe(FIL__, __LINE__, pipedes) < 0) 
    {
      if (envp[0] != NULL) 
	my_free(envp[0]);
      SL_RETURN( (NULL), _("sh_gpg_popen"));
    }
  
  source->pid = aud_fork(FIL__, __LINE__);
  
  /* Failure
   */
  if (source->pid == (pid_t) - 1) 
    {
      close(pipedes[0]);
      close(pipedes[1]);
      if (envp[0] != NULL) 
	my_free(envp[0]);
      SL_RETURN( (NULL), _("sh_gpg_popen"));
    }

  if (source->pid == (pid_t) 0) 
    {

      /* child - make read side of the pipe stdout 
       */
      if (retry_aud_dup2(FIL__, __LINE__,
			pipedes[STDOUT_FILENO], STDOUT_FILENO) < 0)
	{
	  TPT(((0), FIL__, __LINE__, _("msg=<dup2 on pipe failed>\n")));
	  aud__exit(FIL__, __LINE__, EXIT_FAILURE);
	}
      
      /* close the pipe descriptors 
       */
      close (pipedes[STDIN_FILENO]);
      close (pipedes[STDOUT_FILENO]);
      

#if defined(WITH_PGP)
      if (mode == 0) 
	{
	  if (retry_aud_dup2(FIL__, __LINE__, fd, STDIN_FILENO) < 0)
	    {
	      TPT(((0), FIL__, __LINE__, _("msg=<dup2 on fd failed>\n")));
	      aud__exit(FIL__, __LINE__, EXIT_FAILURE);
	    }
	}
#else
      if (retry_aud_dup2(FIL__, __LINE__, fd, STDIN_FILENO) < 0)
	{
	  TPT(((0), FIL__, __LINE__, _("msg=<dup2 on fd failed>\n")));
	  aud__exit(FIL__, __LINE__, EXIT_FAILURE);
	}
#endif
 
      /* don't leak file descriptors
       */
      sh_unix_closeall (3);

      freopen(_("/dev/null"), "r+", stderr); 

      /* We should become privileged if SUID,
       * to be able to read the keyring.
       * We have checked that gpg is OK,
       * AND that only a trusted user could overwrite
       * gpg.
       */
      memset (skey, '\0', sizeof(sh_key_t));
      sl_set_suid(); 
      aud_setuid(FIL__, __LINE__, geteuid());
      
      /* exec the program */

#if defined(__linux__) && defined(HAVE_GPG_CHECKSUM)
      /* 
       * --  emulate an fexecve with checksum testing
       */
#if defined(WITH_PGP)
      checkfd = sl_open_read(DEFAULT_PGP_PATH, SL_NOPRIV);
#else
      checkfd = sl_open_read(DEFAULT_GPG_PATH, SL_NOPRIV);
#endif

      if (0 != sh_gpg_checksum(checkfd, 0))
	aud__exit(FIL__, __LINE__, EXIT_FAILURE);

      pfd = get_the_fd(checkfd);
      sprintf(pname, _("/proc/self/fd/%d"),             /* known to fit  */
		   pfd);
      if (0 == access(pname, R_OK|X_OK))
	{
	  fcntl  (pfd, F_SETFD, FD_CLOEXEC);
	  retry_aud_execve (FIL__, __LINE__,  pname, arg, envp);
	      
	  /* failed 
	   */
	  aud__exit(FIL__, __LINE__, EXIT_FAILURE);
	}
	  
      /* procfs not working, go ahead 
       */
      sl_close(checkfd);
#endif

#if defined(HAVE_GPG_CHECKSUM)
      /* This is an incredibly ugly kludge to prevent an attacker
       * from knowing when it is safe to slip in a fake executable
       * between the integrity check and the execve
       */
      myrand = (int) taus_get (&(skey->rng0[0]), &(skey->rng1[0]), 
			       &(skey->rng2[0]));
      myrand = (myrand < 0) ? (-myrand) : myrand;
      myrand = (myrand % 32) + 2;

      for (i = 0; i < myrand; ++i)
	{
#if defined(WITH_PGP)
	  checkfd = sl_open_fastread(DEFAULT_PGP_PATH, SL_NOPRIV);
#else
	  checkfd = sl_open_fastread(DEFAULT_GPG_PATH, SL_NOPRIV);
#endif
	  if (0 != sh_gpg_checksum(checkfd, 0))
	    aud__exit(FIL__, __LINE__, EXIT_FAILURE);
	  sl_close(checkfd);
	}
#endif
			       

#if defined(WITH_GPG)
      retry_aud_execve (FIL__, __LINE__, DEFAULT_GPG_PATH, arg, envp);
#elif defined(WITH_PGP)
      retry_aud_execve (FIL__, __LINE__, DEFAULT_PGP_PATH, arg, envp);
#endif
      
      /* failed 
       */
      TPT(((0), FIL__, __LINE__, _("msg=<execve failed>\n")));
      aud__exit(FIL__, __LINE__, EXIT_FAILURE);
    }

  /* parent
   */

  if (envp[0] != NULL) 
    my_free(envp[0]);

  close (pipedes[STDOUT_FILENO]);
  retry_fcntl (FIL__, __LINE__, pipedes[STDIN_FILENO], F_SETFD, FD_CLOEXEC);
  retry_fcntl (FIL__, __LINE__, pipedes[STDIN_FILENO], F_SETFL,  O_NONBLOCK);

  MBLK( outf = fdopen (pipedes[STDIN_FILENO], "r"); )
  
  if (outf == NULL) 
    {
      aud_kill (FIL__, __LINE__, source->pid, SIGKILL);
      close (pipedes[STDOUT_FILENO]);
      waitpid (source->pid, NULL, 0);
      source->pid = 0;
      SL_RETURN( (NULL), _("sh_gpg_popen"));
    }
  
  SL_RETURN( (outf), _("sh_gpg_popen"));
}


static int sh_gpg_pclose (sh_gpg_popen_t *source)
{
  int status = 0;
  
  SL_ENTER(_("sh_gpg_pclose"));

  MBLK( status = fclose(source->pipe); )
  if (status)
    SL_RETURN( (-1), _("sh_gpg_pclose"));
  
  if (waitpid(source->pid, NULL, 0) != source->pid)
    status = -1;
  
  source->pipe = NULL;
  source->pid = 0;
  SL_RETURN( (status), _("sh_gpg_pclose"));
}
 
static
int sh_gpg_check_file_sign(int fd, char * sign_id, char * sign_fp)
{
  struct stat buf;
  char line[256];
  sh_gpg_popen_t  source;
  int have_id = BAD, have_fp = BAD, status = 0;
#ifdef WITH_PGP
  char *ptr;
#endif

#ifdef HAVE_GPG_CHECKSUM
  SL_TICKET checkfd;
#endif

  SL_ENTER(_("sh_gpg_check_file_sign"));

  /* check whether GnuPG exists and has the correct checksum
   */
#if defined(WITH_GPG)

  TPT(((0), FIL__, __LINE__, _("msg=<Check signature>\n")));
  TPT(((0), FIL__, __LINE__, _("msg=<gpg is %s>\n"), DEFAULT_GPG_PATH));

  if (0 != retry_lstat(FIL__, __LINE__, DEFAULT_GPG_PATH, &buf))
    {
      status = errno;
      sh_error_handle(SH_ERR_ERR, FIL__, __LINE__, status, MSG_ERR_LSTAT,
		      sh_error_message(status), DEFAULT_GPG_PATH);
      SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));
    }
  if (0 != tf_trust_check (DEFAULT_GPG_PATH, SL_YESPRIV))
    SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));

#ifdef HAVE_GPG_CHECKSUM
  checkfd = sl_open_read(DEFAULT_GPG_PATH, SL_YESPRIV);

  if (0 != sh_gpg_checksum(checkfd, 1))
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      _("Checksum mismatch"), 
		      _("gpg_check_file_sign"));
      SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));
    }
#endif

#elif defined(WITH_PGP)

  TPT(((0), FIL__, __LINE__, _("msg=<Check signature>\n")));
  TPT(((0), FIL__, __LINE__, _("msg=<pgp is %s>\n"), DEFAULT_PGP_PATH));

  if (0 != retry_lstat(FIL__, __LINE__, DEFAULT_PGP_PATH, &buf))
    {
      status = errno;
      sh_error_handle(SH_ERR_ERR, FIL__, __LINE__, status, MSG_ERR_LSTAT,
		      sh_error_message(status), DEFAULT_PGP_PATH);
      SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));
    }
  if (0 != tf_trust_check (DEFAULT_PGP_PATH, SL_YESPRIV))
    SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));

#ifdef HAVE_GPG_CHECKSUM
  checkfd = sl_open_read(DEFAULT_PGP_PATH, SL_YESPRIV);

  if (0 != sh_gpg_checksum(checkfd, 1))
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      _("Checksum mismatch"), 
		      _("gpg_check_file_sign"));
      SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));
    }
#endif

#endif

  TPT(((0), FIL__, __LINE__, _("msg=<Open pipe to check signature>\n")));

  fflush(NULL);
 
  source.pipe   = sh_gpg_popen  ( &source, fd, 0, NULL );

  if (NULL == source.pipe)
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      _("Could not open pipe"), 
		      _("gpg_check_file_sign"));
      SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));
    }

  TPT(((0), FIL__, __LINE__, _("msg=<Open pipe success>\n")));

 xagain:

  errno = 0;

  while (NULL != fgets(line, 255, source.pipe))
    {

      if (line[strlen(line)-1] == '\n')
	line[strlen(line)-1] = ' ';
      sh_error_handle(SH_ERR_ALL, FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      line, 
		      _("gpg_check_file_sign"));

      if (sl_strlen(line) < 18) 
	continue;
#if defined(WITH_GPG)
      /* Sun May 27 18:40:05 CEST 2001
       */
      if (0 == sl_strncmp(_("BADSIG"), &line[9], 6) ||
	  0 == sl_strncmp(_("ERRSIG"), &line[9], 6) ||
	  0 == sl_strncmp(_("SIGEXPIRED"), &line[9], 6))
	{
	  have_fp = BAD; have_id = BAD;
	  break;
	}
      if (0 == sl_strncmp(_("GOODSIG"), &line[9], 7))
	{
	  sl_strlcpy (sign_id, &line[25], SH_MINIBUF+1);
	  sign_id[sl_strlen(sign_id)-1] = '\0';  /* remove trailing '"' */
	  have_id = GOOD;
	} 
      if (0 == sl_strncmp(_("VALIDSIG"), &line[9], 8))
	{
	  strncpy (sign_fp, &line[18], 40);
	  sign_fp[40] = '\0';
	  have_fp = GOOD;
	}
#elif defined(WITH_PGP)
      if (0 == sl_strncmp(_("Bad signature"), line, 13) ||
	  0 == sl_strncmp(_("Error"), line, 5) ||
	  0 == sl_strncmp(_("Malformed"), line, 9) ||
	  0 == sl_strncmp(_("WARNING"), line, 7) ||
	  0 == sl_strncmp(_("ERROR"), line, 5) 
	  )
	{
	  have_fp = BAD; have_id = BAD;
	  break;
	}
      if (0 == sl_strncmp(_("Good signature"), line, 14))
	{
	  ptr = strchr ( line, '"');
	  ++ptr;
	  sl_strlcpy (sign_id, ptr, SH_MINIBUF+1);
	  sign_id[sl_strlen(sign_id)-1] = '\0'; /* remove trailing dot */
	  sign_id[sl_strlen(sign_id)-2] = '\0'; /* remove trailing '"' */
	  have_id = GOOD;
	}
#endif
    }

  if (ferror(source.pipe) && errno == EAGAIN) 
    {
      clearerr(source.pipe);
      goto xagain;
    }
 
  sh_gpg_pclose (&source);

  TPT(((0), FIL__, __LINE__, _("msg=<Close pipe>\n")));

#ifdef WITH_PGP
  /* get the fingerprint */

  source.pipe   = sh_gpg_popen  ( &source, fd, 1,  sign_id);
  if (NULL == source.pipe)
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      _("Could not open pipe for fp"), 
		      _("gpg_check_file_sign"));
      SL_RETURN( SH_GPG_BAD, _("sh_gpg_check_file_sign"));
    }

  TPT(((0), FIL__, __LINE__, _("msg=<Open pipe success>\n")));

 yagain:

  errno = 0;

  while (NULL != fgets(line, 255, source.pipe))
    {
      if (line[strlen(line)-1] == '\n')
	line[strlen(line)-1] = ' ';
      sh_error_handle(SH_ERR_ALL, FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      line, 
		      _("gpg_check_file_sign"));

      if (sl_strlen(line) < 18) 
	continue;
      ptr = strtok (line, " ");
      while (ptr)
	{
	  ptr = strtok (NULL, " ");
	  if (ptr && 0 == sl_strncmp (ptr, _("fingerprint"), 11))
	    {
	      ptr = strtok (NULL, " "); /* to '=' */
	      sign_fp[0] = '\0';
	      while (ptr)
		{
		  ptr = strtok (NULL, " "); /* part of fingerprint */
		  sl_strlcat (sign_fp, ptr, SH_MINIBUF+1);
		}
	      /* sign_fp[sl_strlen(sign_fp)-1] = '\0'; remove trailing '\n' */
	      if (sl_strlen(sign_fp) > 0) 
		have_fp = GOOD;
	      break;
	    } 
	} 
    }

  if (ferror(source.pipe) && errno == EAGAIN) 
    {
      clearerr(source.pipe);
      goto yagain;
    }
 
  sh_gpg_pclose (&source);
#endif

  if (have_id == GOOD)
    {
      TPT(((0), FIL__, __LINE__, _("msg=<Got signator ID>\n")));
      ;
    }
  if (have_fp == GOOD)
    {
      TPT(((0), FIL__, __LINE__, _("msg=<Got fingerprint>\n")));
      ;
    }

  if (have_id == GOOD && have_fp == GOOD)
    SL_RETURN( SH_GPG_OK, _("sh_gpg_check_file_sign"));
  else
    {
      if (have_id == BAD)
	sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
			_("No good signature"), 
			_("gpg_check_file_sign"));
      else
	sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
			_("No fingerprint for key"), 
			_("gpg_check_file_sign"));
      SL_RETURN( SH_GPG_BADSIGN, _("sh_gpg_check_file_sign"));
    }
}

int get_the_fd(SL_TICKET file_1);

int sh_gpg_check_sign (long file_1, long file_2, int what)
{
  int status = SH_GPG_BAD;
  int fd1 = 0;
  int fd2 = 0;
  static int smsg = S_FALSE;
  char  * tmp;
  char  * tmp2;

  SL_ENTER(_("sh_gpg_check_sign"));

  if (what == 0 || what == 1)
    fd1 = get_the_fd(file_1);
  if (what == 0 || what == 2)
    fd2 = get_the_fd(file_2);

  if (fd1 < 0 || fd2 < 0)
    {
      TPT(((0), FIL__, __LINE__, _("msg=<GPG_CHECK: FD1 = %d>\n"), fd1));
      TPT(((0), FIL__, __LINE__, _("msg=<GPG_CHECK: FD2 = %d>\n"), fd2));
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1, sh.prg_name);
      aud_exit (FIL__, __LINE__, EXIT_FAILURE);
      SL_RETURN( (-1), _("sh_gpg_check_sign"));
    }
  
  if (what == 0 || what == 1)
    {
      TPT(((0), FIL__, __LINE__, _("msg=<GPG_CHECK: FD1 = %d>\n"), fd1));
      status = sh_gpg_check_file_sign (fd1, gp.conf_id, gp.conf_fp);
      TPT(((0), FIL__, __LINE__, _("msg=<CONF SIGUSR: |%s|>\n"), gp.conf_id));
      TPT(((0), FIL__, __LINE__, _("msg=<CONF SIGFP:  |%s|>\n"), gp.conf_fp));
    }

  if ((what == 0 && SH_GPG_OK == status) || what == 2)
    {
      TPT(((0), FIL__, __LINE__, _("msg=<GPG_CHECK: FD2 = %d>\n"), fd2));
      status = sh_gpg_check_file_sign (fd2, gp.data_id, gp.data_fp);
      TPT(((0), FIL__, __LINE__, _("msg=<DATA SIGUSR: |%s|>\n"), gp.data_id));
      TPT(((0), FIL__, __LINE__, _("msg=<DATA SIGFP:  |%s|>\n"), gp.data_fp));
    }
  
  if (SH_GPG_OK == status && what == 1)
    {
#ifdef USE_FINGERPRINT
      if ((sl_strcmp(SH_GPG_FP, gp.conf_fp) == 0))
	{
	  if (smsg == S_FALSE)
	    {
	      tmp  = sh_util_safe_name(gp.conf_id);
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_GH,
			       sh.prg_name, sh.real.uid,
			       (sh.flag.hidefile == S_TRUE) ? 
			       _("(hidden)") : file_path('C', 'R'), 
			       tmp, 
			       gp.conf_fp);
	      SH_FREE(tmp);
	    }
	  smsg = S_TRUE;
	  SL_RETURN(0, _("sh_gpg_check_sign"));
	}
      else
	{
	  /* fp mismatch
	   */
	  sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN, 
		      _("Fingerprint mismatch"), 
		      _("gpg_check_sign"));
	  status = SH_GPG_BADSIGN;
	}
#else
      if (smsg == S_FALSE)
	{
	  tmp = sh_util_safe_name(gp.conf_id);
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_GH,
			   sh.prg_name, sh.real.uid,
			   (sh.flag.hidefile == S_TRUE) ? 
			   _("(hidden)") : file_path('C', 'R'), 
			   tmp, 
			   gp.conf_fp);
	  SH_FREE(tmp);
	}
      smsg = S_TRUE;
      SL_RETURN(0, _("sh_gpg_check_sign"));
#endif
    }
  
  else if (SH_GPG_OK == status && (what == 2 || what == 0))
    {
      if ((sl_strcmp(gp.data_id, gp.conf_id) == 0) &&
	  (sl_strcmp(gp.data_fp, gp.conf_fp) == 0))
	{
	  SL_RETURN(0, _("sh_gpg_check_sign"));
	}
      else
	{
	  /* ID or fp not equal 
	   */
	  tmp  = sh_util_safe_name (gp.conf_id);
	  tmp2 = sh_util_safe_name (gp.data_id);
	  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_GH2,
			   sh.prg_name, sh.real.uid,
			   (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('C', 'R'),
			   tmp,  gp.conf_fp,
			   (sh.flag.hidefile == S_TRUE) ? _("(hidden)") : file_path('D', 'R'),
			   tmp2, gp.data_fp);
	  SH_FREE(tmp);
	  SH_FREE(tmp2);
	}
    }


  TPT(((0), FIL__, __LINE__, _("msg=<Status = %d>\n"), status));

  sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_EXIT_ABORT1, sh.prg_name);
  aud_exit (FIL__, __LINE__, EXIT_FAILURE);

  return (-1); /* make compiler happy */
}  

/* #ifdef WITH_GPG */
#endif








