/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2001 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>


#ifdef SH_USE_SUIDCHK

#undef  FIL__
#define FIL__  _("sh_suidchk.c")

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#define NAMLEN(dirent) sl_strlen((dirent)->d_name)
#else
#define dirent direct
#define NAMLEN(dirent) (dirent)->d_namlen
#ifdef HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif
#ifdef HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif
#ifdef HAVE_NDIR_H
#include <ndir.h>
#endif
#endif

#include "samhain.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_modules.h"
#include "sh_suidchk.h"
#include "sh_hash.h"
#include "sh_unix.h"
#include "sh_files.h"


sh_rconf sh_suidchk_table[] = {
  {
    N_("severitysuidcheck"),
    sh_suidchk_set_severity
  },
  {
    N_("suidcheckactive"),
    sh_suidchk_set_activate
  },
  {
    N_("suidcheckinterval"),
    sh_suidchk_set_timer
  },
  {
    N_("suidcheckexclude"),
    sh_suidchk_set_exclude
  },
  {
    N_("suidcheckfps"),
    sh_suidchk_set_fps
  },
  {
    NULL,
    NULL
  },
};


static time_t  lastcheck         = (time_t) 0;
static int     ShSuidchkActive   = S_TRUE;
static long    ShSuidchkInterval = 7200;
static long    ShSuidchkFps      = 0;
static int     ShSuidchkSeverity = SH_ERR_SEVERE;
static char *  ShSuidchkExclude  = NULL;
static int     ExcludeLen        = 0;

static time_t  FileLimNow        = 0;
static time_t  FileLimStart      = 0;
static long    FileLimNum        = 0;
static long    FileLimTotal      = 0;

static char *
filesystem_type (char * path, char * relpath, struct stat * statp);

#ifndef PATH_MAX
#define PATH_MAX 1023
#endif

extern unsigned long sh_files_maskof (int class);


int sh_suidchk_check_internal (char * iname)
{
  DIR *           thisDir = NULL;
  struct dirent * thisEntry;
  char          * tmpcat;
  char          * tmp;
  struct stat     buf;
  int             status;
  char          * fs;
  long            sl_status;

  file_type       theFile;
  char            fileHash[KEY_LEN + 1];

  SL_ENTER(_("sh_suidchk_check_internal"));

  if (iname == NULL)
    {
      TPT((0, FIL__, __LINE__ , _("msg=<directory name is NULL>\n")));
      SL_RETURN( (-1), _("sh_suidchk_check_internal"));
    }

  if (sig_terminate == 1)
    SL_RETURN( (0), _("sh_suidchk_check_internal"));

  sl_set_suid();
  thisDir = opendir (iname);
  sl_unset_suid();

  if (thisDir == NULL)
    {
      status = errno;
      tmp = sh_util_safe_name(iname);
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_OPENDIR,
		       sh_error_message(status),
		       tmp );
      SH_FREE(tmp);
      SL_RETURN( (-1), _("sh_suidchk_check_internal"));
    }

  do {

    if (sig_terminate == 1)
      SL_RETURN( (0), _("sh_suidchk_check_internal"));


    thisEntry = readdir (thisDir);

    if (thisEntry != NULL) {

      if (sl_strcmp (thisEntry->d_name, ".") == 0)
	continue;

      if (sl_strcmp (thisEntry->d_name, "..") == 0)
	continue;

      tmpcat = SH_ALLOC(PATH_MAX+1);
      sl_status = sl_strlcpy(tmpcat, iname,                 PATH_MAX+1);
      if (! SL_ISERROR(sl_status) && tmpcat[1] != '\0')
	sl_status = sl_strlcat(tmpcat, "/",                 PATH_MAX+1);
      if (! SL_ISERROR(sl_status))
	sl_status = sl_strlcat(tmpcat, thisEntry->d_name,   PATH_MAX+1);
      if (SL_ISERROR(sl_status))
	{
	  tmp = sh_util_safe_name(tmpcat);
	  sh_error_handle ((-1), FIL__, __LINE__, (int) sl_status, 
			   MSG_E_SUBGPATH,
			   _("path too long"),
			   _("sh_suidchk_check_internal"), tmp );
	  SH_FREE(tmp);
	  continue;
	}

      ++FileLimNum;
      ++FileLimTotal;

      if (ShSuidchkFps > 0 && FileLimNum > ShSuidchkFps && FileLimTotal > 0)
	{
	  FileLimNum  = 0;
	  FileLimNow  = time(NULL);
 
	  if ( (FileLimNow  - FileLimStart) > 0 && 
	       FileLimTotal/(FileLimNow  - FileLimStart) > ShSuidchkFps )
	    sleep( (int)(FileLimTotal/(FileLimNow  - FileLimStart)) /
		   ShSuidchkFps );
	}
	      
      sl_set_suid();
      status = retry_lstat(FIL__, __LINE__, tmpcat, &buf);
      sl_unset_suid();
      if (status != 0)
	{
	  status = errno;
	  tmp = sh_util_safe_name(tmpcat);
	  sh_error_handle (SH_ERR_ERR, FIL__, __LINE__, status, MSG_ERR_LSTAT,
			   sh_error_message(status),
			   tmpcat );
	  SH_FREE(tmp);
	}
      else
	{
	  if (S_ISDIR(buf.st_mode) &&
	      (ShSuidchkExclude == NULL || 
	      0 != strncmp(tmpcat, ShSuidchkExclude, ExcludeLen)))
	    {
	      /* fs is a STATIC string
	       */
	      fs = filesystem_type (tmpcat, tmpcat, &buf);
	      if (fs != NULL && 
		  0 != strncmp (_("nfs"),     fs, 3) && 
		  0 != strncmp (_("proc"),    fs, 4) &&
		  0 != strncmp (_("iso9660"), fs, 7) &&
		  0 != strncmp (_("vfat"),    fs, 4) &&
		  0 != strncmp (_("msdos"),   fs, 5) &&
		  0 != strncmp (_("devfs"),   fs, 5)
		  )
		{
		  /* fprintf(stderr, "%s: %s\n", fs, tmpcat); */
		  sh_suidchk_check_internal(tmpcat);
		}
	    }
	  else if (S_ISREG(buf.st_mode) &&
		   (0 !=(S_ISUID & buf.st_mode)||0 !=(S_ISGID & buf.st_mode)))
	    {
	      
	      sl_strlcpy (theFile.fullpath, tmpcat, PATH_MAX+1);
	      theFile.check_mask = sh_files_maskof(SH_LEVEL_READONLY);
	      theFile.reported   = S_FALSE;
	      status = sh_unix_getinfo (ShDFLevel[SH_ERR_T_RO], 
					thisEntry->d_name,
					&theFile, fileHash);

	      tmp = sh_util_safe_name(tmpcat);

	      if (status != 0)
		{
		  sh_error_handle (ShSuidchkSeverity, FIL__, __LINE__, 
				   0, MSG_SUID_POLICY,
				   _("Could not check suid/sgid file"),
				   tmp );
		}
	      else
		{
		  if (sh.flag.checkSum == SH_CHECK_INIT  && 
		      sh.flag.update == S_TRUE )
		    {
		      sh_hash_compdata (SH_LEVEL_READONLY, &theFile, fileHash);
		      sh_hash_pushdata (&theFile, fileHash);
		      sh_error_handle ((-1), FIL__, __LINE__, 
				       0, MSG_SUID_FOUND,
				       tmp );
		    }
		  else if (sh.flag.checkSum == SH_CHECK_INIT  && 
			   sh.flag.update == S_FALSE )
		    {
		      sh_hash_pushdata (&theFile, fileHash);
		      sh_error_handle ((-1), FIL__, __LINE__, 
				       0, MSG_SUID_FOUND,
				       tmp );
		    }
		  else if (sh.flag.checkSum == SH_CHECK_CHECK )
		    {
		      sh_error_handle (SH_ERR_ALL, FIL__, __LINE__, 
				       0, MSG_SUID_FOUND,
				       tmp );
		      if (-1 == sh_hash_have_it (tmpcat))
			sh_error_handle (ShSuidchkSeverity, FIL__, __LINE__, 
					 0, MSG_SUID_POLICY,
					 _("suid/sgid file not in database"),
					 tmp );
		      else
			sh_hash_compdata (SH_LEVEL_READONLY, 
					  &theFile, fileHash);
		    }
		}
	      SH_FREE(tmp);

	    }
	}
      SH_FREE(tmpcat);
    }
  }  while (thisEntry != NULL);

  closedir (thisDir);
  SL_RETURN( (0), _("sh_suidchk_check_internal"));
}

/*************
 *
 * module init
 *
 *************/
int sh_suidchk_init ()
{
  if (ShSuidchkActive == S_FALSE)
    return (-1);

  return (0);
}

/*************
 *
 * module cleanup
 *
 *************/
int sh_suidchk_end ()
{
  return (0);
}


/*************
 *
 * module timer
 *
 *************/
int sh_suidchk_timer (unsigned long tcurrent)
{
  if ((time_t) (tcurrent - lastcheck) >= ShSuidchkInterval)
    {
      lastcheck  = tcurrent;
      return (-1);
    }
  return 0;
}

/*************
 *
 * module check
 *
 *************/

int sh_suidchk_check ()
{
  int status;

  SL_ENTER(_("sh_suidchk_check"));

  sh_error_handle (SH_ERR_INFO, FIL__, __LINE__, EINVAL, MSG_E_SUBGEN,
		   _("Checking for SUID programs"),
		   _("suidchk_check") );

  FileLimNow        = time(NULL);
  FileLimStart      = FileLimNow;
  FileLimNum        = 0;
  FileLimTotal      = 0;

  status = sh_suidchk_check_internal ("/");

  sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_SUID_SUMMARY,
		   FileLimTotal,
		   (long) (time(NULL) - FileLimStart) );

  SL_RETURN(status, _("sh_suidchk_check"));
}

/*************
 *
 * module setup
 *
 *************/

int sh_suidchk_set_severity  (char * c)
{
  char tmp[32];

  SL_ENTER(_("sh_suidchk_set_severity"));
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShSuidchkSeverity);
  SL_RETURN(0, _("sh_suidchk_set_severity"));
}

int sh_suidchk_set_exclude (char * c)
{
  SL_ENTER(_("sh_suidchk_set_exclude"));
  if (c == NULL)
    return -1;

  if (ShSuidchkExclude != NULL)
    SH_FREE(ShSuidchkExclude);
  ShSuidchkExclude = sh_util_strdup (c);
  ExcludeLen       = sl_strlen (ShSuidchkExclude);
  SL_RETURN(0, _("sh_suidchk_set_exclude"));
}

int sh_suidchk_set_timer (char * c)
{
  long val;

  SL_ENTER(_("sh_suidchk_set_timer"));

  val = strtol (c, (char **)NULL, 10);
  if (val <= 0)
    sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("suidchk timer"), c);

  val = (val <= 0 ? 7200 : val);

  ShSuidchkInterval = (time_t) val;
  SL_RETURN( 0, _("sh_suidchk_set_timer"));
}

int sh_suidchk_set_fps (char * c)
{
  long val;

  SL_ENTER(_("sh_suidchk_set_fps"));

  val = strtol (c, (char **)NULL, 10);
  if (val < 0)
    sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("suidchk fps"), c);

  val = (val < 0 ? 0 : val);

  ShSuidchkFps = val;
  SL_RETURN( 0, _("sh_suidchk_set_fps"));
}

int sh_suidchk_set_activate (char * c)
{
  int i;
  SL_ENTER(_("sh_suidchk_set_activate"));
  i = sh_util_flagval(c, &ShSuidchkActive);
  SL_RETURN(i, _("sh_suidchk_set_activate"));
}

#ifdef FSTYPE_STATFS
/* dirname.c -- return all but the last element in a path
   Copyright (C) 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Return the leading directories part of PATH,
   allocated with malloc.  If out of memory, return 0.
   Assumes that trailing slashes have already been
   removed.  */

char *
dirname (path)
     char *path;
{
  char *newpath;
  char *slash;
  int length;                   /* Length of result, not including NUL.  */

  slash = strrchr (path, '/');
  if (slash == 0)
    {
      /* File is in the current directory.  */
      path = ".";
      length = 1;
    }
  else
    {
      /* Remove any trailing slashes from the result.  */
      while (slash > path && *slash == '/')
        --slash;

      length = slash - path + 1;
    }
  newpath = (char *) SH_ALLOC (length + 1);
  if (newpath == 0)
    return 0;
  strncpy (newpath, path, length);
  newpath[length] = 0;
  return newpath;
}
/* #ifdef FSTYPE_STATFS */
#endif

/* fstype.c -- determine type of filesystems that files are on
   Copyright (C) 1990, 91, 92, 93, 94 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by David MacKenzie <djm@gnu.ai.mit.edu>. */

/* Modified by R. Wichmann: 
   - replaced error()   by sh_error_handle()
   - replaced xstrdup() by sl_strdup()
   - replaced strstr()  by sl_strstr()
*/

/* modetype.h -- file type bits definitions for POSIX systems
   Requires sys/types.h sys/stat.h.
   Copyright (C) 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* POSIX.1 doesn't mention the S_IFMT bits; instead, it uses S_IStype
   test macros.  To make storing file types more convenient, define
   them; the values don't need to correspond to what the kernel uses,
   because of the way we use them. */
#ifndef S_IFMT			/* Doesn't have traditional Unix macros. */
#define S_IFBLK 1
#define S_IFCHR 2
#define S_IFDIR 4
#define S_IFREG 8
#ifdef S_ISLNK
#define S_IFLNK 16
#endif
#ifdef S_ISFIFO
#define S_IFIFO 32
#endif
#ifdef S_ISSOCK
#define S_IFSOCK 64
#endif
#endif /* !S_IFMT */

#ifdef STAT_MACROS_BROKEN
#undef S_ISBLK
#undef S_ISCHR
#undef S_ISDIR
#undef S_ISREG
#undef S_ISFIFO
#undef S_ISLNK
#undef S_ISSOCK
#undef S_ISMPB
#undef S_ISMPC
#undef S_ISNWK
#endif

/* Do the reverse: define the POSIX.1 macros for traditional Unix systems
   that don't have them.  */
#if !defined(S_ISBLK) && defined(S_IFBLK)
#define	S_ISBLK(m) (((m) & S_IFMT) == S_IFBLK)
#endif
#if !defined(S_ISCHR) && defined(S_IFCHR)
#define	S_ISCHR(m) (((m) & S_IFMT) == S_IFCHR)
#endif
#if !defined(S_ISDIR) && defined(S_IFDIR)
#define	S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
#endif
#if !defined(S_ISREG) && defined(S_IFREG)
#define	S_ISREG(m) (((m) & S_IFMT) == S_IFREG)
#endif
#if !defined(S_ISFIFO) && defined(S_IFIFO)
#define	S_ISFIFO(m) (((m) & S_IFMT) == S_IFIFO)
#endif
#if !defined(S_ISLNK) && defined(S_IFLNK)
#define	S_ISLNK(m) (((m) & S_IFMT) == S_IFLNK)
#endif
#if !defined(S_ISSOCK) && defined(S_IFSOCK)
#define	S_ISSOCK(m) (((m) & S_IFMT) == S_IFSOCK)
#endif
#if !defined(S_ISMPB) && defined(S_IFMPB) /* V7 */
#define S_ISMPB(m) (((m) & S_IFMT) == S_IFMPB)
#define S_ISMPC(m) (((m) & S_IFMT) == S_IFMPC)
#endif
#if !defined(S_ISNWK) && defined(S_IFNWK) /* HP/UX */
#define S_ISNWK(m) (((m) & S_IFMT) == S_IFNWK)
#endif


static char *filesystem_type_uncached (char *path, char *relpath, 
				       struct stat *statp);

#ifdef FSTYPE_MNTENT		/* 4.3BSD etc.  */
static int xatoi (char *cp);
#endif

#ifdef FSTYPE_MNTENT		/* 4.3BSD, SunOS, HP-UX, Dynix, Irix.  */
#include <mntent.h>
#if !defined(MOUNTED)
# if defined(MNT_MNTTAB)	/* HP-UX.  */
#  define MOUNTED MNT_MNTTAB
# endif
# if defined(MNTTABNAME)	/* Dynix.  */
#  define MOUNTED MNTTABNAME
# endif
#endif
#endif

#ifdef FSTYPE_GETMNT		/* Ultrix.  */
#include <sys/param.h>
#include <sys/mount.h>
#include <sys/fs_types.h>
#endif

#ifdef FSTYPE_USG_STATFS	/* SVR3.  */
#include <sys/statfs.h>
#include <sys/fstyp.h>
#endif

#ifdef FSTYPE_STATVFS		/* SVR4.  */
#include <sys/statvfs.h>
#include <sys/fstyp.h>
#endif

#ifdef FSTYPE_STATFS		/* 4.4BSD.  */
#include <sys/param.h>		/* NetBSD needs this.  */
#include <sys/mount.h>

#ifndef MFSNAMELEN		/* NetBSD defines this.  */
static char *
fstype_to_string (t)
     short t;
{
#ifdef INITMOUNTNAMES		/* Defined in 4.4BSD, not in NET/2.  */
  static char *mn[] = INITMOUNTNAMES;
  if (t >= 0 && t <= MOUNT_MAXTYPE)
    return mn[t];
  else
    return "?";
#else /* !INITMOUNTNAMES */
  switch (t)
    {
    case MOUNT_UFS:
      return _("ufs");
    case MOUNT_NFS:
      return _("nfs");
#ifdef MOUNT_PC
    case MOUNT_PC:
      return _("pc");
#endif
#ifdef MOUNT_MFS
    case MOUNT_MFS:
      return _("mfs");
#endif
#ifdef MOUNT_LO
    case MOUNT_LO:
      return _("lofs");
#endif
#ifdef MOUNT_TFS
    case MOUNT_TFS:
      return _("tfs");
#endif
#ifdef MOUNT_TMP
    case MOUNT_TMP:
      return _("tmp");
#endif
#ifdef MOUNT_MSDOS
    case MOUNT_MSDOS:
      return _("msdos");
#endif
#ifdef MOUNT_ISO9660
    case MOUNT_ISO9660:
      return _("iso9660fs");
#endif
    default:
      return "?";
    }
#endif /* !INITMOUNTNAMES */
}
#endif /* !MFSNAMELEN */
#endif /* FSTYPE_STATFS */

#ifdef FSTYPE_AIX_STATFS	/* AIX.  */
#include <sys/vmount.h>
#include <sys/statfs.h>

#define FSTYPE_STATFS		/* Otherwise like 4.4BSD.  */
#define f_type f_vfstype

static char *
fstype_to_string (t)
     short t;
{
  switch (t)
    {
    case MNT_AIX:
#if 0				/* NFS filesystems are actually MNT_AIX. */
      return "aix";
#endif
    case MNT_NFS:
      return _("nfs");
    case MNT_JFS:
      return _("jfs");
    case MNT_CDROM:
      return _("cdrom");
    default:
      return "?";
    }
}
#endif /* FSTYPE_AIX_STATFS */

#ifdef AFS
#include <netinet/in.h>
#include <afs/venus.h>
#if __STDC__
/* On SunOS 4, afs/vice.h defines this to rely on a pre-ANSI cpp.  */
#undef _VICEIOCTL
#define _VICEIOCTL(id)  ((unsigned int ) _IOW('V', id, struct ViceIoctl))
#endif
#ifndef _IOW
/* AFS on Solaris 2.3 doesn't get this definition.  */
#include <sys/ioccom.h>
#endif

static int
in_afs (path)
     char *path;
{
  static char space[2048];
  struct ViceIoctl vi;

  vi.in_size = 0;
  vi.out_size = sizeof (space);
  vi.out = space;

  if (pioctl (path, VIOC_FILE_CELL_NAME, &vi, 1)
      && (errno == EINVAL || errno == ENOENT))
	return 0;
  return 1;
}
#endif /* AFS */

/* Nonzero if the current filesystem's type is known.  */
static int fstype_known = 0;

/* Return a static string naming the type of filesystem that the file PATH,
   described by STATP, is on.
   RELPATH is the file name relative to the current directory.
   Return "unknown" if its filesystem type is unknown.  */

static char *
filesystem_type (char * path, char * relpath, struct stat * statp)
{
  static char *current_fstype = NULL;
  static dev_t current_dev;

  if (current_fstype != NULL)
    {
      if (fstype_known && statp->st_dev == current_dev)
	return current_fstype;	/* Cached value.  */
      SH_FREE (current_fstype);
    }
  current_dev = statp->st_dev;
  current_fstype = filesystem_type_uncached (path, relpath, statp);
  return current_fstype;
}

/* Return a newly allocated string naming the type of filesystem that the
   file PATH, described by STATP, is on.
   RELPATH is the file name relative to the current directory.
   Return "unknown" if its filesystem type is unknown.  */

static char *
filesystem_type_uncached (path, relpath, statp)
     char *path;
     char *relpath;
     struct stat *statp;
{
  char * type = NULL;
#ifdef MFSNAMELEN		/* NetBSD.  */
  static char my_tmp_type[64];
#endif

#ifdef FSTYPE_MNTENT		/* 4.3BSD, SunOS, HP-UX, Dynix, Irix.  */
  char *table = MOUNTED;
  FILE *mfp;
  struct mntent *mnt;

  if (path == NULL || relpath == NULL)
    return NULL;

  mfp = setmntent (table, "r");
  if (mfp == NULL)
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
		       _("setmntent() failed"),
		       _("filesystem_type_uncached") );
      return NULL;
    }

  /* Find the entry with the same device number as STATP, and return
     that entry's fstype. */
  while (type == NULL && (mnt = getmntent (mfp)))
    {
      char *devopt;
      dev_t dev;
      struct stat disk_stats;

#ifdef MNTTYPE_IGNORE
      if (!strcmp (mnt->mnt_type, MNTTYPE_IGNORE))
	continue;
#endif

      /* Newer systems like SunOS 4.1 keep the dev number in the mtab,
	 in the options string.	 For older systems, we need to stat the
	 directory that the filesystem is mounted on to get it.

	 Unfortunately, the HPUX 9.x mnttab entries created by automountq
	 contain a dev= option but the option value does not match the
	 st_dev value of the file (maybe the lower 16 bits match?).  */

#if !defined(hpux) && !defined(__hpux__)
      devopt = sl_strstr (mnt->mnt_opts, "dev=");
      if (devopt)
	{
	  if (devopt[4] == '0' && (devopt[5] == 'x' || devopt[5] == 'X'))
	    dev = xatoi (devopt + 6);
	  else
	    dev = xatoi (devopt + 4);
	}
      else
#endif /* not hpux */
	{
	  if (stat (mnt->mnt_dir, &disk_stats) == -1)
	    {
	      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
			       _("stat() failed"),
			       _("filesystem_type_uncached") );
	      return NULL;
	    }
	  dev = disk_stats.st_dev;
	}

      if (dev == statp->st_dev)
	type = mnt->mnt_type;
    }

  if (endmntent (mfp) == 0)
    {
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_E_SUBGEN,
		       _("endmntent() failed"),
		       _("filesystem_type_uncached") );
    }
#endif

#ifdef FSTYPE_GETMNT		/* Ultrix.  */
  int offset = 0;
  struct fs_data fsd;

  if (path == NULL || relpath == NULL)
    return NULL;

  while (type == NULL
	 && getmnt (&offset, &fsd, sizeof (fsd), NOSTAT_MANY, 0) > 0)
    {
      if (fsd.fd_req.dev == statp->st_dev)
	type = gt_names[fsd.fd_req.fstype];
    }
#endif

#ifdef FSTYPE_USG_STATFS	/* SVR3.  */
  struct statfs fss;
  char typebuf[FSTYPSZ];

  if (path == NULL || relpath == NULL)
    return NULL;

  if (statfs (relpath, &fss, sizeof (struct statfs), 0) == -1)
    {
      /* Don't die if a file was just removed. */
      if (errno != ENOENT)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, errno, MSG_E_SUBGEN,
			   _("statfs() failed"),
			   _("filesystem_type_uncached") );
	  return NULL;
	}
    }
  else if (!sysfs (GETFSTYP, fss.f_fstyp, typebuf))
    type = typebuf;
#endif

#ifdef FSTYPE_STATVFS		/* SVR4.  */
  struct statvfs fss;

  if (path == NULL || relpath == NULL)
    return NULL;

  if (statvfs (relpath, &fss) == -1)
    {
      /* Don't die if a file was just removed. */
      if (errno != ENOENT)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, errno, MSG_E_SUBGEN,
			   _("statvfs() failed"),
			   _("filesystem_type_uncached") );
	  return NULL;
	}
    }
  else
    type = fss.f_basetype;
#endif

#ifdef FSTYPE_STATFS		/* 4.4BSD.  */
  struct statfs fss;
  char *p;
  /* extern char * dirname(const char *path); */

  if (path == NULL || relpath == NULL)
    return NULL;

  if (S_ISLNK (statp->st_mode))
    p = dirname (relpath);
  else
    p = relpath;

  if (statfs (p, &fss) == -1)
    {
      /* Don't die if symlink to nonexisting file, or a file that was
	 just removed. */
      if (errno != ENOENT)
	{
	  sh_error_handle ((-1), FIL__, __LINE__, errno, MSG_E_SUBGEN,
			   _("statfs() failed"),
			   _("filesystem_type_uncached") );
	  return NULL;
	}
    }
  else
    {
#ifdef MFSNAMELEN		/* NetBSD.  */
      /* MEMORY LEAK !!!
       *	 type = sh_util_strdup (fss.f_fstypename);
       */
      sl_strlcpy (my_tmp_type, fss.f_fstypename, 64);
      type = my_tmp_type;
#else
      type = fstype_to_string (fss.f_type);
#endif
    }
  if (p != relpath)
    SH_FREE (p);
#endif

#ifdef AFS
  if ((!type || !strcmp (type, "xx")) && in_afs (relpath))
    type = "afs";
#endif

  /* An unknown value can be caused by an ENOENT error condition.
     Don't cache those values.  */
  fstype_known = (type != NULL);

  return sh_util_strdup (type ? type : "unknown");
}

#ifdef FSTYPE_MNTENT		/* 4.3BSD etc.  */
/* Return the value of the hexadecimal number represented by CP.
   No prefix (like '0x') or suffix (like 'h') is expected to be
   part of CP. */

static int
xatoi (cp)
     char *cp;
{
  int val;
  
  val = 0;
  while (*cp)
    {
      if (*cp >= 'a' && *cp <= 'f')
	val = val * 16 + *cp - 'a' + 10;
      else if (*cp >= 'A' && *cp <= 'F')
	val = val * 16 + *cp - 'A' + 10;
      else if (*cp >= '0' && *cp <= '9')
	val = val * 16 + *cp - '0';
      else
	break;
      cp++;
    }
  return val;
}
#endif



#endif


/* #ifdef SH_USE_UTMP */
#endif



