/* switch off full check */
/* #define TEST_ONLY     */

/* debug problems        */
/* #define TRUST_DEBUG   */

/* LINTLIBRARY */
/*
 * This is the file with all the library routines in it
 *
 * Author information:
 * Matt Bishop
 * Department of Computer Science
 * University of California at Davis
 * Davis, CA  95616-8562
 * phone (916) 752-8060
 * email bishop@cs.ucdavis.edu
 *
 * This code is placed in the public domain.  I do ask that
 * you keep my name associated with it, that you not represent
 * it as written by you, and that you preserve these comments.
 * This software is provided "as is" and without any guarantees
 * of any sort.
 *
 * Compilation notes:
 * * this does NOT use malloc(3), but fixed storage.  this means we
 *   do lots of bounds checking, but it still is faster, and smaller,
 *   than forcing inclusion of malloc.  All buffers etc. are of size
 *   MAXFILENAME (defined in trustfile.h); to get more room, recompile
 *   with this set larger.
 * * if you support the following directory semantics, define STICKY;
 *   otherwise, undefine it
 *	"if a directory is both world-writeable AND has the sticky bit
 *	 set, then ONLY the owner of an existing file may delete it"
 *   On some systems (eg, IRIX), you can delete the file under these
 *   conditions if the file is world writeable.  Foor our purposes,
 *   this is irrelevant since if the file is world-writeable it is
 *   untrustworthy; either it can be replaced with another file (the
 *   IRIX version) or it can be altered (all versions).
 *   if this is true and STICKY is not set, the sticky bit is ignored
 *   and the directory will be flagged as untrustworthy, even when only
 *   a trusted user could delete the file
 * * this uses a library call to get the name of the current working
 *   directory.  Define the following to get the various versions:
 *   GETCWD	for Solaris 2.x, SunOS 4.1.x, IRIX 5.x
 *			char *getcwd(char *buf, int bufsz);
 *		where buf is a buffer for the path name, and bufsz is
 *		the size of the buffer; if the size if too small, you
 *		get an error return (NULL)
 *   GETWD	for Ultrix 4.4
 *			char *getwd(char *buf)
 *		where buf is a buffer for the path name, and it is
 *		assumed to be at lease as big as MAXPATHLEN.
 *		*** IMPORTANT NOTE ***
 *		Ultrix supports getcwd as well, but it uses popen to
 *		run the command "pwd" (according to the manual).  This
 *		means it's vulnerable to a number of attacks if used
 *		in a privileged program.  YOU DON'T WANT THIS.
 * * the debugging flag DEBUG prints out each step of the file name
 *   checking, as well as info on symbolic links (if S_IFLNK defined),
 *   file name canonicalization, and user, group, and permission for
 *   each file or directory; this is useful if you want to be sure
 *   you're checking the right file
 *
 * Version information:
 * 1.0		December 28, 1995	Matt Bishop
 *
 * 2.0          March    26, 2000       Rainer Wichmann -- adapted for slib.
 */

/* --- Why strcpy is safe here: ----                                  */

/* The input path is checked once, and then either shortened [in dirz()], 
 * or safely expanded (symlinks) with bound checking.
 * I.e., the path length can never exceed (MAXFILENAME-1), while the path
 * is always copied between buffers of length MAXFILENAME.
 */

#include "config_xor.h"
#include "sh_calls.h"


#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#include <string.h>
#include <unistd.h>

/* -- Defined in config.h. --

#ifndef _(string)
#define _(string) string
#endif

#ifndef N_(string)
#define N_(string) string
#endif
*/

#include "slib.h"


#if defined(__linux__)
#define STICKY
#endif

#undef  FIL__
#define FIL__  _("trustfile.c")

/*
 * the get current working directory function
 * every version of UNIX seems to have its own
 * idea of how to do this, so we group them by
 * arguments ...
 * all must return a pointer to the right name
 */


#if defined(HAVE_GETCWD) && !defined(HAVE_BROKEN_GETCWD)
#define CURDIR(buf,nbuf)	getcwd((buf), (nbuf))
#elif defined(HAVE_GETWD)
#define CURDIR(buf,nbuf)	getwd((buf))
#endif



/*
 * this checks to see if there are symbolic links
 * assumes the link bit in the protection mask is called S_IFLNK
 * (seems to be true on all UNIXes with them)
 */
#ifndef S_IFLNK
#define	lstat	stat
#endif


/*
 * these are useful global variables
 *
 * first set: who you gonna trust, by default?
 * 	if the user does not specify a trusted or untrusted set of users,
 *	all users are considered untrusted EXCEPT:
 *	UID 0 -- root	as root can do anything on most UNIX systems, this
 *			seems reasonable
 *	tf_euid -- programmer-selectable UID
 *			if the caller specifies a specific UID by putting
 *			it in this variable, it will be trusted; this is
 *			typically used to trust the effective UID of the
 *			process (note: NOT the real UID, which will cause all
 *			sorts of problems!)  By default, this is set to -1,
 *			so if it's not set, root is the only trusted user
 */

/* modified Tue Feb 22 10:36:44 NFT 2000 Rainer Wichmann                */


#ifndef SL_ALWAYS_TRUSTED
#define SL_ALWAYS_TRUSTED  0
#endif
static long test_rootonly[] = { SL_ALWAYS_TRUSTED };

long rootonly[] = { SL_ALWAYS_TRUSTED, 
		    -1, -1, -1, -1, -1, -1, -1, 
		    -1, -1, -1, -1, -1, -1, -1, -1, -1};
long tf_euid = -1;
int EUIDSLOT = sizeof(test_rootonly)/sizeof(long);
int ORIG_EUIDSLOT = sizeof(test_rootonly)/sizeof(long);

char tf_path[MAXFILENAME];		/* error path for trust function */

static 
int dirz(char *path)
{
  register char *p = path;/* points to rest of path to clean up */
  register char *q;	/* temp pointer for skipping over stuff */

  char swp[MAXFILENAME];
  SL_ENTER(_("dirz"));
  /*
   * standard error checking
   */
  if (path == NULL)
    SL_IRETURN(SL_ENULL, _("dirz"));
  if (path[0] == '.')
    SL_IRETURN(SL_EINTERNAL, _("dirz"));
  
  
  /*
   * loop over the path name until everything is checked
   */
  while(*p)
    {
      /* skip 
       */
      if (*p != '/')
	{
	  p++;
	  continue;
	}

      /* "/./" or "/." 
       */
      if (p[1] == '.' && (p[2] == '/' || p[2] == '\0'))
	{
	  /* yes -- delete "/." 
	   */
	  (void) strcpy(swp, &p[2]);                     /* known to fit  */
	  (void) strcpy(p, swp);                         /* known to fit  */

	  /* special case "/." as full path name 
	   */
	  if (p == path && *p == '\0')
	    {
	    *p++ = '/';
	    *p = '\0';
	  }
      }

      /* "//" 
       */
      else if (p[1] == '/')
	{
	  /* yes -- skip 
	   */
	  for(q = &p[2]; *q == '/'; q++)
	    ;
	  (void) strcpy(swp, q);                         /* known to fit  */
	  (void) strcpy(&p[1], swp);                     /* known to fit  */
	}

      /* "/../" or "/.." 
       */
      else if (p[1] == '.' && p[2] == '.' && (p[3] == '/' || p[3] == '\0'))
	{
	  /* yes -- if it's root, delete .. only 
	   */
	  if (p == path)
	    {
	      (void) strcpy(swp, &p[3]);                 /* known to fit  */
	      (void) strcpy(p, swp);                     /* known to fit  */
	    }
	  else
	    {
	      /* back up over previous component 
	       */
	      q = p - 1;
	      while(q != path && *q != '/')
		q--;
	      /* now wipe it out 
	       */
	      (void) strcpy(swp, &p[3]);                 /* known to fit  */
	      (void) strcpy(q, swp);                     /* known to fit  */
	      p = q;
	    }
	}
      else
	p++;
    }
  SL_IRETURN(SL_ENONE, _("dirz"));
}
			


static 
int getfname(char *fname, char *rbuf, int rsz)
{
  register int status;
  
  SL_ENTER(_("getfname"));
  /*
   * do the initial checking
   * NULL pointer
   */
  if (fname == NULL || rbuf == NULL)
    SL_IRETURN(SL_ENULL, _("getfname"));
  if (rsz <= 0)
    SL_IRETURN(SL_ERANGE, _("getfname"));
  
  
  /* already a full path name */
  if (*fname == '/')
    rbuf[0] = '\0';
  else
    {
      if (CURDIR(rbuf, rsz)  == NULL)
	SL_IRETURN(SL_EBADNAME, _("getfname"));
    }
  
  /*
   * append the file name and reduce
   */
  if (fname != NULL && *fname != '\0')
    {
      status = sl_strlcat(rbuf, "/", rsz);
      if (status == SL_ENONE)
	status = sl_strlcat(rbuf, fname, rsz);
      if (status != SL_ENONE)
	SL_IRETURN(status, _("getfname"));
    }
  SL_IRETURN(dirz(rbuf), _("getfname"));
}

static 
int isin(long n, long *list)
{
  SL_ENTER(_("isin"));
  if (list == NULL)
    SL_IRETURN(SL_FALSE, _("isin"));

  while(*list != -1 && *list != n)
    {
#ifdef TRUST_DEBUG
      fprintf (stderr, "FIXME: n=%ld, *list=%ld\n", n, *list);
#endif 
      list++;
    }

  if (*list == -1)
    SL_IRETURN(SL_FALSE, _("isin"));

  SL_IRETURN(SL_TRUE, _("isin"));
}

/* comment added by R. Wichmann
 *  RETURN TRUE if ANYONE in ulist is group member
 */
static 
int isingrp(long grp, long *ulist)
{
  register struct passwd *w;	/* info about group member */
  register long *u;		/* points to current ulist member */
  register char **p;		/* points to current group member */
  register struct group *g;	/* pointer to group information */
  
  SL_ENTER(_("isingrp"));

  if ((g = getgrgid(grp)) == NULL)
    SL_IRETURN(SL_FALSE, _("isingrp") );

  /*
  if(g->gr_mem == NULL || g->gr_mem[0] == NULL )
    SL_IRETURN(SL_FALSE, _("isingrp") );
  */

  /* this will return at the first match
   */
  for(p = g->gr_mem; *p != NULL; p++)
    for(u = ulist; *u != -1; u++)
      /* map user name to UID and compare */
      if ((w = getpwnam(*p)) != NULL && *u == (long)(w->pw_uid) )
	SL_IRETURN(SL_TRUE, _("isingrp"));

  /* added by R. Wichmann Fri Mar 30 08:16:14 CEST 2001: 
   * a user can have a GID but no entry in /etc/group
   */
  for(u = ulist; *u != -1; u++)
    if ((w = getpwuid(*u)) != NULL && grp == (long)(w->pw_gid) )
      SL_IRETURN(SL_TRUE, _("isingrp"));

  SL_IRETURN(SL_FALSE, _("isingrp"));
}

/* added by R. Wichmann Fri Mar 30 08:16:14 CEST 2001
 *  RETURN TRUE only if ALL group members are trusted
 */
static 
int onlytrustedingrp(long grp, long *ulist)
{
  register struct passwd *w;	/* info about group member */
  register long *u;		/* points to current ulist member */
  register char **p;		/* points to current group member */
  register struct group *g;	/* pointer to group information */
  register int flag = -1;       /* group member found */

  SL_ENTER(_("onlytrustedingrp"));

#ifdef TRUST_DEBUG
      fprintf(stderr, "GROUP: %ld\n", grp); 
#endif

  if ((g = getgrgid(grp)) == NULL)
    SL_IRETURN(SL_FALSE, _("onlytrustedingrp") );

  /* empty group -> no problem
   
  if(g->gr_mem == NULL || g->gr_mem[0] == NULL )
    SL_IRETURN(SL_TRUE, _("onlytrustedingrp") );
  */

  /* check for untrusted members of the group
   */
  for(p = g->gr_mem; *p != NULL; p++)
    {
      flag = -1;
#ifdef TRUST_DEBUG
      fprintf(stderr, "GROUP MEMBER: %s\n", *p); 
#endif
      /* map user name to UID and compare 
       */
      w = getpwnam(*p);
      if (w == NULL)    /* not a valid user, ignore    */
	{
	  flag = 0; 
	}
      else              /* check list of trusted users */
	{
	  for(u = ulist; *u != -1; u++)
	    {
	      if (*u == (long)(w->pw_uid) )
		flag = 0;
	    }
	}
      /* not found 
       */
      if (flag == -1)
	  SL_IRETURN(SL_FALSE, _("onlytrustedingrp"));
    }

#ifndef TEST_ONLY	
#ifdef HAVE_GETPWENT
  /* now check ALL users for their GID !!!
   */
  while (NULL != (w = getpwent())) 
    {
      if (grp == (long)(w->pw_gid))
	{
#ifdef TRUST_DEBUG
	  fprintf(stderr, "USER: %s %d\n", w->pw_name, (int)w->pw_uid); 
#endif
	  /* is it a trusted user ?
	   */
	  flag = -1;
	  for(u = ulist; *u != -1; u++)
	    {
	      if (*u == (long)(w->pw_uid))
		flag = 0;
	    }
	  /* not found */
	  if (flag == -1)
	    {
#ifdef TRUST_DEBUG
	      fprintf(stderr, "USER: not found in trusted users\n"); 
#endif
	      SL_IRETURN(SL_FALSE, _("onlytrustedingrp"));
	    }
	}
    }

#ifdef HAVE_ENDPWENT
  endpwent();
#endif
  /* TEST_ONLY */
#endif

  /* #ifdef HAVE_GETPWENT */
#endif

  /* all found
   */
  SL_IRETURN(SL_TRUE, _("onlytrustedingrp"));
}

int sl_trustfile(char *fname, long *okusers, long *badusers)
{
  char fexp[MAXFILENAME];	/* file name fully expanded        */
  register char *p = fexp;      /* used to hold name to be checked */
  struct stat stbuf;	        /* used to check file permissions  */
  char c;			/* used to hold temp char          */
  
  SL_ENTER(_("sl_trustfile"));
  if (fname == NULL)
    SL_IRETURN(SL_EBADFILE, _("sl_trustfile"));

  /*
   * next expand to the full file name
   * getfname sets sl_errno as appropriate
   */
  if (SL_ISERROR(getfname(fname, fexp, MAXFILENAME)))
      SL_IRETURN(sl_errno, _("sl_trustfile"));

  if (okusers == NULL && badusers == NULL)
    {
      okusers = rootonly;
      rootonly[EUIDSLOT] = tf_euid;
    }

  /*
   * now loop through the path a component at a time
   * note we have to special-case root
   */
  while(*p)
    {
      /*
       * get next component
       */
      while(*p && *p != '/')
	p++;

      /* save where you are 
       */
      if (p == fexp)
	{
	  /* keep the / if it's the root dir 
	   */
	  c    = p[1];
	  p[1] = '\0';
	}
      else
	{
	  /* clobber the / if it isn't the root dir 
	   */
	  c  = *p;
	  *p = '\0';
	}

      /*
       * now get the information
       */
      if (retry_lstat(FIL__, __LINE__, fexp, &stbuf) < 0)
	{
	  (void) strcpy(tf_path, fexp);                  /* known to fit  */
	  SL_IRETURN(SL_ESTAT, _("sl_trustfile"));
	}

#ifdef S_IFLNK
      /* 
       * if it's a symbolic link, recurse
       */
      if ((stbuf.st_mode & S_IFLNK) == S_IFLNK)
	{
	  /*
	   * this is tricky
	   * if the symlink is to an absolute path
	   * name, just call trustfile on it; but
	   * if it's a relative path name, it's 
	   * interpreted WRT the current working
	   * directory AND NOT THE FILE NAME!!!!!
	   * so, we simply put /../ at the end of
	   * the file name, then append the symlink
	   * contents; trustfile will canonicalize
	   * this, and the /../ we added "undoes"
	   * the name of the symlink to put us in
	   * the current working directory, at
	   * which point the symlink contents (appended
	   * to the CWD) are interpreted correctly.
	   * got it?
	   */
	  char csym[MAXFILENAME];	/* contents of symlink file  */
	  char full[MAXFILENAME];	/* "full" name of symlink    */
	  register char *b, *t;	        /* used to copy stuff around */
	  register int lsym;	        /* num chars in symlink ref  */
	  register int i;		/* trustworthy or not?       */

	  /*
	   * get what the symbolic link points to
	   *
	   * The original code does not check the return code of readlink(),
	   * and has an off-by-one error 
	   * (MAXFILENAME instead of MAXFILENAME-1)
	   * R.W. Tue May 29 22:05:16 CEST 2001
	   */
	  lsym = readlink(fexp, csym, MAXFILENAME-1);
	  if (lsym >= 0) 
	    csym[lsym] = '\0';
	  else
	    SL_IRETURN(SL_EBADNAME, _("sl_trustfile"));

	  /*
	   * relative or absolute referent?
	   */
	  if (csym[0] != '/')
	    {
	      /* initialize pointers 
	       */
	      b = full;

	      /* copy in base path 
	       */
	      t = fexp;
	      while(*t && b < &full[MAXFILENAME])
		*b++ = *t++;

	      /* smack on the /../ 
	       */
	      t = "/../";
	      while(*t && b < &full[MAXFILENAME])
		*b++ = *t++;

	      /* append the symlink referent 
	       */
	      t = csym;
	      while(*t && b < &full[MAXFILENAME])
		*b++ = *t++;

	      /* see if we're too big 
	       */
	      if (*t || b == &full[MAXFILENAME])
		{
		  /* yes -- error 
		   */
		  (void) strcpy(tf_path, fexp);          /* known to fit  */
		  SL_IRETURN(SL_ETRUNC, _("sl_trustfile"));
		}
	      *b = '\0';
	    }
	  else
	    {
	      /* absolute -- just copy                */
	      /* overflow can't occur as the arrays   */
	      /* are the same size		      */
	      (void) strcpy(full, csym);                 /* known to fit  */
	    }
	  /*
	   * now check out this file and its ancestors
	   */
	  if ((i = sl_trustfile(full, okusers, badusers)) != SL_ENONE)
	    SL_IRETURN(i, _("sl_trustfile"));

	  /*
	   * okay, this part is valid ... let's check the rest
	   * put the / back
	   */
	  if (p == fexp)
	    {
	      /* special case for root */
	      p[1] = c;
	      p++;
	    }
	  else
	    {
	      /* ordinary case for everything else */
	      *p = c;
	      if (*p)
		p++;
	    }
	  continue;
	}
#endif

			
      /*
       * if the owner is not trusted then -- as the owner can
       * change protection modes -- he/she can write to the
       * file regardless of permissions, so bomb
       */
      if (((okusers != NULL && SL_FALSE == isin((long)stbuf.st_uid,okusers))||
	   (badusers != NULL && SL_TRUE == isin((long)stbuf.st_uid,badusers))))
	{
#ifdef TRUST_DEBUG
	  fprintf(stderr, "FIXME: EBADUID %s\n", fexp); 
#endif 
	  (void) strcpy(tf_path, fexp);                  /* known to fit  */
	  SL_IRETURN(SL_EBADUID, _("sl_trustfile"));
	}

      /*
       * if a group member can write but the
       * member is not trusted, bomb; but if
       * sticky bit semantics are honored, it's
       * okay
       */
      /* Thu Mar 29 21:10:28 CEST 2001 Rainer Wichmann
       * replace !isingrp() with onlytrustedingrp(), as isingrp()
       * will return at the first trusted user, even if there are additional
       * (untrusted) users in the group
       */
      if (((stbuf.st_mode & S_IWGRP) == S_IWGRP) &&
	  ((okusers != NULL && !onlytrustedingrp((long)stbuf.st_gid,okusers))||
	   (badusers != NULL && isingrp((long)stbuf.st_gid, badusers)))
#ifdef STICKY
	  && ((stbuf.st_mode&S_IFDIR) != S_IFDIR ||
	      (stbuf.st_mode&S_ISVTX) != S_ISVTX)
#endif
	  )
	{
#ifdef TRUST_DEBUG
	  fprintf(stderr, "FIXME: EBADGID %d %s\n", (int)stbuf.st_gid, fexp);
#endif 
	  (void) strcpy(tf_path, fexp);                  /* known to fit  */
	  SL_IRETURN(SL_EBADGID, _("sl_trustfile"));
	}
      /*
       * if other can write, bomb; but if the sticky
       * bit semantics are honored, it's okay
       */
      if (((stbuf.st_mode & S_IWOTH) == S_IWOTH)
#ifdef STICKY
	  && ((stbuf.st_mode&S_IFDIR) != S_IFDIR ||
	      (stbuf.st_mode&S_ISVTX) != S_ISVTX)
#endif
	  )
	{
	  (void) strcpy(tf_path, fexp);                  /* known to fit  */
	  SL_IRETURN(SL_EBADOTH, _("sl_trustfile"));
	}
      /*
       * put the / back
       */
      if (p == fexp)
	{
	  /* special case for root */
	  p[1] = c;
	  p++;
	}
      else
	{
	  /* ordinary case for everything else */
	  *p = c;
	  if (*p)
	    p++;
	}
    }
  /*
   * yes, it can be trusted
   */
  (void) strcpy(tf_path, fexp);                      /* known to fit  */
  SL_IRETURN(SL_ENONE, _("sl_trustfile"));
}

