/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2000 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <string.h>

#include "samhain.h"
#include "sh_error.h"
#include "sh_utils.h"
#include "sh_tiger.h"

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif


#undef  FIL__
#define FIL__  _("sh_err_log.c")

#define MYSIGLEN (2*KEY_LEN + 32)

typedef struct _sh_log_buf {
  char   signature[KEY_LEN+1];
  char   timestamp[KEY_LEN+1];
#ifdef SH_USE_XML
  char   sig[MYSIGLEN];
#endif
  char * msg;
} sh_sh_log_buf;

extern struct  _errFlags  errFlags;

#define CHK_KEY 0
#define CHK_FIL 1
#define CHK_NON 2

int get_key_from_file(char * path, char * keyid, char * key)
{
  SL_TICKET  fd;
  char * buf;
  char * bufc;

  if (path[strlen(path)-1] == '\n')
    path[strlen(path)-1] = '\0';

  /* open the file, then check it 
   */
  if ( SL_ISERROR(fd = sl_open_read (path, SL_NOPRIV)))
    {
      fprintf(stderr, _("Could not open file <%s>\n"), path);
      _exit (EXIT_FAILURE);
    }

  buf     = SH_ALLOC( (unsigned long)(SH_BUFSIZE+1));
  bufc    = SH_ALLOC( (unsigned long)(SH_MAXBUF+1));

  while (1)
    {
      buf[0]  = '\0';
      bufc[0] = '\0';

      /* find start of next key
       */
      while (0 != sl_strncmp(buf, _("-----BEGIN LOGKEY-----"),
			     sizeof("-----BEGIN LOGKEY-----")-1)) 
	{
	  (void) sh_unix_getline (fd, buf, SH_BUFSIZE);
	  if (buf[0] == '\0')
	    {
	      /* End of file reached, return. 
	       */
	      fflush(stdout);
	      sl_close(fd);
	      return -1; 
	    }
	}

      /* read key
       */
      (void) sh_unix_getline (fd, buf, SH_BUFSIZE);

      if (0 == sl_strncmp(keyid, &buf[KEY_LEN], strlen(keyid)))
	{
	  sl_strlcpy(key, buf, KEY_LEN+1);
	  sl_close(fd);
	  return 0;
	}
    }
	  
  /* not found
   */
  sl_close(fd);
  return -1;
}

static int just_list = S_FALSE;

int sh_error_logverify_mod (char * s)
{
  just_list = S_TRUE;
  if (s)      /* compiler warning (unused var) fix */
    return 0;
  else
    return 0;
} 

int sh_error_logverify (char * s)
{
  SL_TICKET fd;
  int len;
  int status;
  int count =  0;
  int start = -1;
  char * buf;
  char * bufc;
#ifdef SH_USE_XML
  char * ptr;
#endif
  char signature[64];
  char key[KEY_LEN+1];
  char path[KEY_LEN+1];
  char timestamp[64];
  char c_cont;
  int  chk_mode = CHK_KEY;

  sh_error_logoff();

  if (s == NULL || sl_strlen(s) >= PATH_MAX)
    {
      fprintf(stderr, _("FAIL: msg=\"Invalid input\", path=\"%s\"\n"), s);
      _exit (EXIT_FAILURE);
    }

  /* Open the file, then check it. 
   */
  if ( SL_ISERROR(fd = sl_open_read (s, SL_NOPRIV)) )
    {
      fprintf(stderr, 
	      _("FAIL: msg=\"File not accessible\", path=\"%s\"\n"), s);
      _exit (EXIT_FAILURE);
    }

  /* Find space value.
   */
  c_cont  = ' ';
#ifdef SH_STEALTH
  c_cont ^= XOR_CODE;
#endif

  buf  = (char *) SH_ALLOC( 2*SH_BUFSIZE+1 );
  bufc = (char *) SH_ALLOC( 2*SH_BUFSIZE+1 );

  while (1) 
    {
      /* get the log message
       */
      if (sh_unix_getline (fd, buf, (2*SH_BUFSIZE)) < 0) 
	break;

      len = (int) sl_strlen(buf);

#ifdef SH_USE_XML
#ifdef SH_STEALTH
      if (0 == sl_strncmp (buf, N_("<trail>"), 7)) 
#else
      if (0 == sl_strncmp (buf, _("<trail>"),  7)) 
#endif
#else 
#ifdef SH_STEALTH
      if (0 == sl_strncmp (buf, N_("[SOF]"), 5)) 
#else
      if (0 == sl_strncmp (buf, _("[SOF]"),  5)) 
#endif
#endif
	{
	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	    }

	  /* Found start of audit trail, read first line. 
	   */
	  start = 1;
	  do {
	    if ( sh_unix_getline (fd, buf, (2*SH_BUFSIZE)) < 0)
	      break;
	  } while (buf[0] == '\0' || buf[0] == '\n');
	  len = (int) sl_strlen(buf);

	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      if (buf[0] != '\n') 
		sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	      start = 0;
	    }

	  ++count;
	}
      else if (buf[0] == '\n'
#ifdef SH_USE_XML
	       ||
#ifdef SH_STEALTH
	       0 == sl_strncmp(buf, N_("</trail>"), 7)
#else
	       0 == sl_strncmp(buf,  _("</trail>"), 7)
#endif
#endif
	       )
	{
	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      if (buf[0] != '\n') 
		sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	    }

	  /* A newline.
	   */
	  ++count;
	  continue;
	}
      else if (start == 0)
	{
	  /* We are inside an audit trail. 
	   */
	  ++count;
	  if (just_list == S_TRUE)
	    {
#ifdef SH_STEALTH
	      sh_do_decode (buf, sl_strlen(buf));
#endif
	      fprintf (stdout, _("%s\n"), buf);
	      continue;
	    }
	}
      else
	{
	  /* No start-of-file found yet. 
	   */
	  continue;
	}

      if (just_list == S_TRUE)
	continue;

      /* Check for a continuation line.
       */
      while (1 == 1)
	{
	  do {
	    if ( sh_unix_getline (fd, bufc, (2*SH_BUFSIZE)) < 0)
	      break;
	  } while (bufc[0] == '\0' || bufc[0] == '\n');
	  ++count;
	  if (bufc[0] == c_cont) 
	    {
	      /* A continuation line. Add the newline. 
	       */
	      sl_strlcat(buf, "\n", 2*SH_BUFSIZE+1);
	      ++len;
	      sl_strlcat(buf, bufc, 2*SH_BUFSIZE+1);
	      len += (int) sl_strlen(bufc);
	    }
	  else
	    {
	      /* No continuation line. Use it as signature. 
	       * A48014C05604EF7C9472330E85453E704024943E556163C2
	       */
#ifdef SH_USE_XML
	      sl_strlcpy(signature, &bufc[4], KEY_LEN+1);
	      sl_strlcpy(signature, &bufc[4], KEY_LEN+1);
	      if (sl_strlen(bufc) > (KEY_LEN+18))
		{
		  sl_strlcpy(timestamp, &bufc[KEY_LEN+4], 64);
#ifdef SH_STEALTH
		  ptr = strchr(timestamp, '<' ^ XOR_CODE);
#else
		  ptr = strchr(timestamp, '<');
#endif
		  if (ptr) *ptr = '\0';
		}
	      break;
#else
	      sl_strlcpy(signature, bufc, KEY_LEN+1);
	      if (sl_strlen(bufc) > KEY_LEN)
		sl_strlcpy(timestamp, &bufc[KEY_LEN], 64);
	      break;
#endif
	    }
	}
      
      /* Get starting key from command line. 
       */    
      if (start == 1) 
	{
	  
	  /* Get the timestamp.
	   */
	  
#ifdef SH_STEALTH
	  sh_do_decode (timestamp, sl_strlen(timestamp));
#endif
	  key[0] = '\0';
	  
	findKey:
	  
	  if (chk_mode != CHK_FIL)
	    {
	      /* Ask for the key.
	       */
	      chk_mode = CHK_KEY;
	      fprintf(stdout, _("\nNew audit trail (%s), enter key|keyfile: "),
		      timestamp);
	      key[0] = '\0';
	      
	      while (sl_strlen(key) < KEY_LEN ) 
		{ 
		  if (key[0] != '\n' && key[0] != '\0')
		    fprintf(stdout, _("New audit trail, enter key: "));
		  else if (key[0] == '\n')
		    {
		      sl_strlcpy(key, sh_tiger_hash(NULL, TIGER_DATA, 0), 
				 KEY_LEN+1);
		      chk_mode = CHK_NON;
		      break;
		    }
		  fflush(stdout); 
		  key[0] = '\0';
		  fgets(key, KEY_LEN+1, stdin);
		  if (key[0] == '/')
		    {
		      chk_mode = CHK_FIL;
		      sl_strlcpy(path, key, KEY_LEN+1); 
		      break;
		    }
		}
	    }
	  /* we now have either a key (chk_mode == CHK_NON|CHK_KEY)
	   * or a file (chk_mode == CHK_FIL)
	   */
	  if (chk_mode == CHK_FIL)
	    {
	      fprintf(stdout, _("\nAudit trail (%s), searching file %s\n"), 
		      timestamp, path);
	      if (-1 == get_key_from_file(path, timestamp, key))
		{
		  chk_mode = CHK_KEY;
		  fprintf(stdout, _("Key not found in file\n"));
		  goto findKey;
		}
	    }
	  
	  
	  sh_util_encode(key, buf, 1, 'B');
	  start = 0;
	} 
      else
	{ 
	  /* Iterate the key.
	   */
	  sl_strlcpy (key, 
		      sh_tiger_hash (key, TIGER_DATA, KEY_LEN), 
		      KEY_LEN+1);
	}
      
      sl_strlcat ( buf, key, 2*SH_BUFSIZE + 1);
      
#ifdef SH_STEALTH
      sh_do_decode (signature, sl_strlen(signature));
#endif
      
      status = sl_strncmp (signature, 
			   sh_tiger_hash (buf, TIGER_DATA, sl_strlen(buf)),
			   KEY_LEN);
      
      buf[len] = '\0';    /* do not print out the key */
#ifdef SH_STEALTH
      sh_do_decode (buf, sl_strlen(buf));
#endif
      
      if (status != 0) 
	{
#ifdef SH_USE_XML
	  if (chk_mode == CHK_NON)
	    fprintf (stdout, _("XFAIL: line=%05d %s/log>\n"), count-1, buf);
	  else
	    fprintf (stdout, _("FAIL:  line=%05d %s/log>\n"), count-1, buf);
#else
	  if (chk_mode == CHK_NON)
	    fprintf (stdout, _("XFAIL: line=%5d %s\n"), count-1, buf);
	  else
	    fprintf (stdout, _("FAIL:  line=%5d %s\n"), count-1, buf);
#endif
	}
      else
	{
#ifdef SH_USE_XML 
	  fprintf (stdout, _("PASS:  line=%05d %s/log>\n"), count-1, buf);
#else
	  fprintf (stdout, _("PASS:  line=%5d %s\n"), count-1, buf);
#endif    
	}
    }

  /* Cleanup and exit.
   */
  sl_close (fd);
  SH_FREE  (buf);
  SH_FREE  (bufc);
  fflush   (stdout);
  _exit    (EXIT_SUCCESS);

  /* Make compilers happy. 
   */
  return 0; 
}

/********************************************************************
 *
 *  Runtime code
 *
 ********************************************************************/

/*
 *   --- Log error message to log file. ---
 */
int  sh_log_file (char *errmsg)
{
  int                  store1;
  int                  store2;
  SL_TICKET            fd = -1;
  long int             status;
  char               * tmp = NULL;
  struct _sh_log_buf   log_msg;
  uid_t                uid;
  static int           service_failure = 0;


  SL_ENTER(_("sh_log_file"));

  if (errFlags.HaveLog == BAD)  /* paranoia */ 
    SL_RETURN((-1), _("sh_log_file"));

  /* open/create the file, then check it 
   */
  tmp  = sh_util_safe_name (sh.srvlog.name);

  if (  0 !=  (status = tf_trust_check (sh.srvlog.name, SL_YESPRIV))
	&& service_failure == 0)
    sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_TRUST,
		      (long) sh.effective.uid, tmp);

  if (status == 0)
    {
      fd = sl_open_write (sh.srvlog.name, SL_YESPRIV);
      if (SL_ISERROR(fd))
	{
	  sl_get_euid(&uid);
	  if (service_failure == 0)
	    sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_ACCESS,
			     (long) uid, tmp);
	  status = -1;
	}
    }


  if (status == 0)
    {
      status = sh_unix_testlock(sh.srvlog.name);
      if (status < 0)
	{
	  sl_get_euid(&uid);
	  if (service_failure == 0)
	    sh_error_handle ((-1), FIL__, __LINE__, status, MSG_LOCKED,
			     (long) uid, tmp, sh.srvlog.alt);
	  status = -1;
	  sl_close(fd);
	}
    }

  if (status == 0)
    {
      status = SL_ISERROR(sl_forward(fd) ); 
      if (status < 0)
	{
	  sl_get_euid(&uid);
	  if (service_failure == 0)
	    sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_ACCESS,
			     (long) uid, tmp);
	  status = -1;
	  sl_close(fd);
	}
    }
  
  if (status < 0)
    {
      if (service_failure == 0) {
	sh_error_handle ((-1), FIL__, __LINE__, status, MSG_SRV_FAIL,
			 _("logfile"), tmp);
	service_failure = 1;
      }
      SH_FREE(tmp);
      SL_RETURN ((-1), _("sh_log_file"));
    }

  service_failure = 0;

#ifdef SH_USE_XML
  if (NULL == errmsg)
    {
#ifdef SH_STEALTH
      sl_write_line (fd, N_("</trail>"), 7);
      sl_write (fd, "\n", 1);
      sl_sync(fd);
#else
      sl_write_line (fd, _("</trail>\n"),  8);
      sl_sync(fd);
#endif
      sl_close(fd);
      SL_RETURN( 0, _("sh_log_file"));
    }
#endif

  /* --- Allocate storage and mlock it. ---
   */

  status      = sl_strlen (errmsg);
  log_msg.msg = (char *) SH_ALLOC ( 2*KEY_LEN + status + 32); 

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
  if (skey->mlock_failed == GOOD) 
    {
      sl_set_suid ();
      if ( (-1) == mlock( log_msg.msg, 2*KEY_LEN + status + 32 ) ) 
	{
	  sl_unset_suid ();
	  skey->mlock_failed = BAD;
	  sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);
	}
      else
	sl_unset_suid ();
    }
#else
  if (skey->mlock_failed == GOOD) 
    {
      skey->mlock_failed = BAD;
      sh_error_handle ((-1), FIL__, __LINE__, EPERM, MSG_MLOCK);
    }
#endif

  /* --- Write the start marker. --- 
   */

  if (sh.flag.log_start == S_TRUE) 
    {
#ifdef SH_USE_XML
#ifdef SH_STEALTH
      sl_write (fd, "\n", 1);
      sl_write_line (fd, N_("<trail>"), 7);
      sl_sync(fd);
#else
      sl_write_line (fd, _("\n<trail>"),  8);
      sl_sync(fd);
#endif
#else
#ifdef SH_STEALTH
      sl_write (fd, "\n", 1);
      sl_write_line (fd, N_("[SOF]"), 5);
      sl_sync(fd);
#else
      sl_write_line (fd, _("\n[SOF]"),  6);
      sl_sync(fd);
#endif
#endif
    }

  /* reserve KEY_LEN chars at end for key 
   */
  sl_strlcpy (log_msg.msg, errmsg, status+1 );

#ifdef SH_USE_XML
  /* cut the trailing "/>"
   */
  if (log_msg.msg[status-2] == '/')
    {
      log_msg.msg[status-2] = '>'; /* ' ' FIX XML */
      log_msg.msg[status-1] = '<'; /* '>' FIX XML */
      log_msg.msg[status]   = '\0';
    }
  else if (log_msg.msg[status-5] == '/')
    {
      log_msg.msg[status-5]   = '\0';
      status -= 5;
    }
#endif

#ifdef SH_STEALTH
  sh_do_encode (log_msg.msg, status);
#endif

  /* write the signature 
   */
  if (sh.flag.log_start == S_TRUE) 
    {
      store1               = errFlags.loglevel;
      store2               = errFlags.sysloglevel;
      errFlags.loglevel    = SH_ERR_NOT;
      errFlags.sysloglevel = SH_ERR_NOT;

      if (sh.real.user[0] == '\0') 
	(void) sh_unix_getUser();

      /* Initialize the key.
       */
      sh_util_keyinit(skey->sigkey_old, KEY_LEN+1);

      /* Hash the key to make sure it has the correct format.
       */
      sl_strlcpy(skey->sigkey_new, 
		 sh_tiger_hash (skey->sigkey_old, TIGER_DATA, KEY_LEN), 
		 KEY_LEN+1);

      /* Copy it to 'crypt' for encryption.
       */
      sl_strlcpy(skey->crypt, skey->sigkey_new, KEY_LEN+1);

      /* Use message and compiled-in key to encrypt.
       */
      BREAKEXIT(sh_util_encode);
      sh_util_encode(skey->crypt, log_msg.msg, 0, 'B');

      /* Send out the key.
       */
      sl_strlcpy(log_msg.timestamp, sh_unix_time(0), KEY_LEN+1); 
      sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_START_KEY,
		       sh.prg_name, skey->crypt, 
		       skey->crypt, log_msg.timestamp);

      /* Cleanup.
       */
      memset (skey->crypt, '\0', KEY_LEN);
      errFlags.loglevel    = store1;
      errFlags.sysloglevel = store2;
      sh.flag.log_start   = S_FALSE;  
    } 
  else 
    {
      log_msg.timestamp[0] = '\0';
      sl_strlcpy (skey->sigkey_new, 
		  sh_tiger_hash (skey->sigkey_old, TIGER_DATA, KEY_LEN),
		  KEY_LEN+1);
    }

  /* --- Sign the message with the signature key. ---
   */
  sl_strlcat (log_msg.msg, skey->sigkey_new, status + KEY_LEN + 2);
  
  sl_strlcpy (log_msg.signature,
	      sh_tiger_hash (log_msg.msg, TIGER_DATA, status + KEY_LEN), 
	      KEY_LEN+1);
  sl_strlcpy (skey->sigkey_old, skey->sigkey_new, KEY_LEN+1); 

#ifdef SH_USE_XML
  if (log_msg.timestamp[0] != '\0')
    sprintf(log_msg.sig,                            /* known to fit  */
	    _("\nsig>%s%s</sig></log>\n"),          /* <sig> FIX XML */
	    log_msg.signature, log_msg.timestamp);
  else
    sprintf(log_msg.sig,                            /* known to fit  */
	    _("\nsig>%s</sig></log>\n"),            /* <sig> FIX XML */
	    log_msg.signature);
#ifdef SH_STEALTH
  /* don't encode the line breaks (0 + last char)
   */
  sh_do_encode (&log_msg.sig[1], (sl_strlen(log_msg.sig)-2) );
#endif
#else
#ifdef SH_STEALTH
  sh_do_encode (log_msg.signature, KEY_LEN);
  sh_do_encode (log_msg.timestamp, sl_strlen(log_msg.timestamp));
#endif
#endif
  
#ifdef SH_USE_XML
  log_msg.msg[status] = '\0';
  sl_strlcat (log_msg.msg,   log_msg.sig, status + 2*KEY_LEN + 32);
#ifdef SH_STEALTH
  if (NULL != sl_strstr(log_msg.msg, N_("EXIT")))
    {
      sl_strlcat (log_msg.msg,  N_("</trail>"), status + 2*KEY_LEN + 32); 
#else
  if (NULL != sl_strstr(log_msg.msg,  _("msg=\"EXIT\"")))
    {
      sl_strlcat (log_msg.msg,   _("</trail>"), status + 2*KEY_LEN + 32); 
#endif
      sl_strlcat (log_msg.msg,   _("\n"), status + 2*KEY_LEN + 32); 
    }
#else
  log_msg.msg[status] = '\0';
  sl_strlcat (log_msg.msg,              "\n", status + KEY_LEN + 2);
  sl_strlcat (log_msg.msg, log_msg.signature, status + KEY_LEN + 2);
  if (log_msg.timestamp[0] != '\0')
    sl_strlcat (log_msg.msg, log_msg.timestamp, status + 2*KEY_LEN + 2);
  sl_strlcat (log_msg.msg,              "\n", status + 2*KEY_LEN + 3);
#endif
  
  /* --- Write out the record. ---
   */
  sl_write (fd, log_msg.msg, strlen(log_msg.msg));
  sl_sync  (fd);
  sl_close (fd);

  /* --- Clean up and free record. ---
   */
  memset (log_msg.msg,       '\0', status + 2*KEY_LEN + 3);
  memset (log_msg.signature, '\0', KEY_LEN);
  MUNLOCK(log_msg.msg,  status + 2*KEY_LEN + 3);
  SH_FREE(log_msg.msg);

  if (tmp != NULL)
    SH_FREE(tmp);
  SL_RETURN (0, _("sh_log_file"));
}

