/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 1999, 2000 Rainer Wichmann                                */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include "config_xor.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <setjmp.h>

#if defined(SH_WITH_MAIL)

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif


#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#include "samhain.h"
#include "sh_error.h"
#include "sh_unix.h"
#include "sh_tiger.h"
#include "sh_mail.h"
#include "sh_utils.h"
#include "sh_fifo.h"
#include "sh_tools.h"

#undef  FIL__
#define FIL__  _("sh_mail.c")

static int failedMail = GOOD;



/*********************************************
 *  utility function for verifying mails
 *********************************************/

typedef struct mail_trail_struct {
  char                     trail_id[2*SH_MINIBUF];
  char                     trail_key[KEY_LEN+1];
  struct mail_trail_struct * next;
} mail_trail_type;

static mail_trail_type * mail_trail = NULL;

int sh_mail_sigverify (char * s)
{
  SL_TICKET  fd;
  int    i;
  char * buf;
  char * bufc;
  char   key[81];
  char   number[2*SH_MINIBUF];
  char   audit_id[2 * SH_MINIBUF];
  long   numsig;
  char   key2[KEY_LEN+1];

  char * theSig;

  mail_trail_type * mail_trail_ptr = NULL;

  sh_error_logoff();

  ASSERT((s != NULL && sl_strlen(s) < PATH_MAX), 
	 _("(s != NULL && sl_strlen(s) < PATH_MAX)"));

  if (s == NULL || sl_strlen(s) >= PATH_MAX) 
    _exit (EXIT_FAILURE);

  /* open the file, then check it 
   */
  if ( SL_ISERROR(fd = sl_open_read (s, SL_NOPRIV)))
    {
      fprintf(stderr, _("Could not open file %s\n"), s);
      _exit (EXIT_FAILURE);
    }

  buf     = SH_ALLOC( (unsigned long)(SH_BUFSIZE+1));
  bufc    = SH_ALLOC( (unsigned long)(SH_MAXBUF+1));

  while (1)
    {
      buf[0]  = '\0';
      bufc[0] = '\0';

      /* find start of next message
       */
      while (0 != sl_strncmp(buf, _("-----BEGIN MESSAGE-----"),
			     sizeof("-----BEGIN MESSAGE-----")-1)) 
	{
	  (void) sh_unix_getline (fd, buf, SH_BUFSIZE);
	  if (buf[0] == '\0')
	    {
	      /* End of mailbox reached, exit. 
	       */
	      fflush(stdout);
	      _exit (EXIT_SUCCESS);

	      /* Fix for AIX cc complaint. 
	       */
	      return 0; 
	    }
	}
      
      /* Read message, compress into bufc.
       */
      while (1)
	{
	  (void) sh_unix_getline (fd, buf, SH_BUFSIZE);
	  if (0 == sl_strncmp(buf, _("-----BEGIN SIGNATURE-----"),
			      sizeof("-----BEGIN SIGNATURE-----")-1))
	    break;
	  if (buf[0] == '\0') 
	    _exit (EXIT_FAILURE);
	  sh_util_compress(bufc, buf, SH_MAXBUF-KEY_LEN);
	}
      
      /* get signature and number 
       */
      (void) sh_unix_getline (fd, key,    sizeof(key)-1);
      key[KEY_LEN] = '\0';

      (void) sh_unix_getline (fd, number, sizeof(number)-1);
      number[(2*SH_MINIBUF) - 2]   = '\0';
      numsig = atol (number);
      sl_strlcpy (audit_id, &number[7], 2*SH_MINIBUF);
      
      fprintf(stderr, _("Message %06ld  Trail %s\n"), 
	      numsig, audit_id);

      mail_trail_ptr = mail_trail;
      while (mail_trail_ptr)
	{
	  if (0 == sl_strcmp(mail_trail_ptr->trail_id, audit_id))
	    break;
	  mail_trail_ptr = mail_trail_ptr->next;
	}

      if (!mail_trail_ptr)
	{
	  if (numsig > 0)
	    {
	      fprintf (stderr, _("ERROR (no key -- cannot check)\n"));
	      continue;
	    }
	  else
	    {
	      mail_trail_ptr = SH_ALLOC (sizeof(mail_trail_type));
	      mail_trail_ptr->next = mail_trail;
	      mail_trail = mail_trail_ptr;
	      sl_strlcpy (mail_trail_ptr->trail_id,  audit_id, 2*SH_MINIBUF);
	    }
	}
      else if (numsig == 0)
	{
	  fprintf (stderr, _("ERROR (repeated audit trail)\n"));
	  continue;
	}
	

      if (numsig == 0)
	{
	  sh_util_encode(key, bufc, 1, 'A');
	  sl_strlcpy (mail_trail_ptr->trail_key,      key,    KEY_LEN+1);
	  fprintf (stderr, _("(unchecked)\n"));
	}
      else
	{
	  /* iterate key
	   */
	  sl_strlcpy(key2, mail_trail_ptr->trail_key, KEY_LEN+1); 
	  for (i = 0; i < numsig; ++i) 
	    {
	      sl_strlcpy (key2, 
		       sh_tiger_hash (key2, TIGER_DATA, KEY_LEN), 
		       KEY_LEN+1);
	    }
	  

	  theSig = sh_util_siggen (key2, bufc, sl_strlen(bufc));
	  if (sl_strncmp (key, 
			  theSig,
			  KEY_LEN) != 0) 
	    {
	      fprintf (stderr, _("(FAILED)\n"));
	    } 
	  else 
	    { 
	      fprintf (stderr, _("(passed)\n"));
	    }

	}

    } /* end scan mailbox */

  /* notreached */
}

static char * address_list[8] = { 
  NULL, NULL, NULL, NULL, 
  NULL, NULL, NULL, NULL 
};

static   int   address_num = 0;
static   int   address_num_compiled = 0;
static   int   setaddress_compiled = S_FALSE;

void reset_count_dev_mail(void)
{
  /* if not, then we still have the compiled-in address (if any), so we
   * don' touch them
   */
  if (address_num_compiled == -99)
    address_num = 0;
  return;
}

int sh_mail_setaddress (char * address)
{
  unsigned long  numcop;
  char     *     p;

  SL_ENTER(_("sh_mail_setaddress"));
  
  if (0 == strcmp(address, _("NULL")))
    SL_RETURN ( (0), _("sh_mail_setaddress"));
    
  if (address != NULL && address_num < (2 * SH_PATHBUF / 64 )) 
    {
      if (address_num < (SH_PATHBUF / 64 ))
	p = &sh.srvmail.name[address_num*64];
      else
	p = &sh.srvmail.alt[address_num*64];

      numcop = sl_strlcpy (p, address, 64);
      
      if (SL_ISERROR(numcop))
	{
	  memset(p, '\0', 64);
	  SL_RETURN ( (-1), _("sh_mail_setaddress"));
	}
      address_list[address_num] = p;
      if (!sl_is_suid())
	{
	  TPT(( 0, FIL__, __LINE__, _("msg=<address_list[%d] = %s>\n"), 
		address_num, address_list[address_num]));
	}
      if (setaddress_compiled == S_TRUE)
	{
	  ++address_num;
	  ++address_num_compiled;
	}
      else
	{
	  if (address_num == address_num_compiled)
	    {
	      address_num = 0;
	      address_num_compiled = -99;
	    }
	  ++address_num;
	}
      SL_RETURN ( (0), _("sh_mail_setaddress"));
    }
  SL_RETURN ( (-1), _("sh_mail_setaddress"));
}

int sh_mail_setaddress_int (char * address)
{
  int i;
  SL_ENTER(_("sh_mail_setaddress_int"));
  setaddress_compiled = S_TRUE;
  i = sh_mail_setaddress(address);
  setaddress_compiled = S_FALSE;
  SL_RETURN(i, _("sh_mail_setaddress_int"));
}
  


int sh_mail_setNum (char * str)
{
  int i = atoi (str);

  SL_ENTER(_("sh_mail_setNum"));

  if (i >= 0 && i < SH_FIFO_MAX) 
    sh.mailNum.alarm_interval = i;
  else 
    SL_RETURN ((-1), _("sh_mail_setNum"));
  SL_RETURN( (0), _("sh_mail_setNum"));
}


static int all_in_one = S_FALSE;

int sh_mail_setFlag (char * str)
{
  int i;
  SL_ENTER(_("sh_mail_setFlag"));
  i = sh_util_flagval(str, &all_in_one);
  SL_RETURN(i, _("sh_mail_setFlag"));
}

static char * mail_subject = NULL;

int set_mail_subject (char * str)
{
  SL_ENTER(_("set_mail_subject"));
  if (!str)
    SL_RETURN( (-1), _("set_mail_subject"));

  if (mail_subject != NULL)
    SH_FREE(mail_subject);

  mail_subject = sh_util_strdup(str);
  SL_RETURN( (0), _("set_mail_subject"));
}


static SH_FIFO * fifo_mail = NULL;

void sh_mail_emptystack (void)
{
  char * msg;
  int    len;

  SL_ENTER(_("sh_mail_emptystack"));

  if (fifo_mail == NULL)
    SL_RET0(_("sh_mail_emptystack"));

  while (NULL != (msg = pop_list(fifo_mail)))
    {
      len = sl_strlen(msg);
      memset(msg, 0, len);
      SH_FREE(msg);
    }

  SL_RET0(_("sh_mail_emptystack"));
}

void sh_mail_pushstack (char * msg)
{
  SL_ENTER(_("sh_mail_pushstack"));

  if (msg == NULL || failedMail == BAD || sh.srvmail.name[0] == '\0') 
    SL_RET0(_("sh_mail_pushstack"));

  if (msg != NULL && sl_strlen(msg) > 998)  /* RFC 2822 */
    msg[998] = '\0';

  if (fifo_mail == NULL)
    {
      fifo_mail = SH_ALLOC(sizeof(SH_FIFO));
      fifo_init(fifo_mail);
    }

  push_list (fifo_mail, msg);
  ++sh.mailNum.alarm_last;

  if (sh.mailNum.alarm_last >= sh.mailNum.alarm_interval)
    {
      BREAKEXIT(sh_mail_msg);
      sh_mail_msg (NULL);
    }

  SL_RET0(_("sh_mail_pushstack"));
}

/* The mailer.
 */
static int sh_mail_end_conn (FILE * connfile);
static FILE * sh_mail_start_conn (int aFlag);

void sh_mail_get_subject(char * message,
			 char * mheader, int len)
{
  st_format rep_serv_tab[] = {
    { 'T', S_FMT_TIME,    0, 0, NULL},
    { 'H', S_FMT_STRING,  0, 0, NULL},
    { 'M', S_FMT_STRING,  0, 0, NULL},
    {'\0', S_FMT_ULONG,   0, 0, NULL},
  };

  char * p;
  char * mptr;

  SL_ENTER(_("sh_mail_get_subject"));

  sl_strlcpy(mheader, _("Subject: "), len);
  if (NULL == strchr(mail_subject, '%'))
    {
      sl_strlcat(mheader, mail_subject, len);
      SL_RET0(_("sh_mail_get_subject"));
    }


  rep_serv_tab[0].data_ulong = (unsigned long) time(NULL);
  rep_serv_tab[1].data_str   = sh.host.name;

  /* fast forward to the important part
   */
  mptr = sl_strstr(message, _("msg="));
  if (mptr)
    {
      mptr += 4;
      rep_serv_tab[2].data_str   = mptr;
    }
  else
    rep_serv_tab[2].data_str   = message;

  p = sh_util_formatted(mail_subject, rep_serv_tab);
  sl_strlcat(mheader, p, len);
  SH_FREE(p);
  SL_RET0(_("sh_mail_get_subject"));
}


#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif

int sh_mail_msg (char * message)
{
    char         subject[32+32+SH_MINIBUF+2+3+SH_PATHBUF];
    char         mheader[32+32+SH_MINIBUF+2+3];

    char       * mailMsg;
    char       * popMsg;
    int          status = 0, errcount;
    int          i;

    char       * bufcompress;
    static int   failcount = 0;
    static int   isfirst   = 1;
    static int   mailcount = 0;
    FILE       * connfile  = NULL;

    struct  sigaction  old_act;
    struct  sigaction  new_act;

    static  time_t id_audit  = 0;
    static  time_t fail_time = 0;
    static  time_t success_time = 0;

    static  int m_block = 0;

    SH_FIFO * fifo_temp = NULL;

    char    * theSig;

    SL_ENTER(_("sh_mail_msg"));

    if (m_block == 1)
      SL_RETURN( (0), _("sh_mail_msg"));

    /* Return if we cannot mail.
     */
    if (failedMail == BAD) 
      SL_RETURN((-1), _("sh_mail_msg"));

    if (failedMail == GOOD && address_list[0] == NULL)
      {
	TPT((0, FIL__, __LINE__, 
	     _("msg=<Mail error: no recipient address.>\n")));
	failedMail = BAD;
	SL_RETURN((-1), _("sh_mail_msg"));
      }

    if (message != NULL && sl_strlen(message) > 998)  /* RFC 2822 */
      message[998] = '\0';

    if ( (success_time > 0) && (fail_time > 0) &&
	 (time(NULL) - success_time) > 3600*SH_MAX_FAIL)
      {
	m_block = 1;
	sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_SRV_FAIL,
			 _("mail"), address_list[0]);
	m_block = 0;
	sh_mail_emptystack();
	failedMail = BAD;
	SL_RETURN((-1), _("sh_mail_msg"));
      }

    /* Try at most each hour.
     */
    if (fail_time > 0 &&  (time(NULL) - fail_time) < 3600)
      {
	/* -- Save for later. -- 
	 */
	sh_mail_pushstack (message);
	++failcount;

	SL_RETURN((-1), _("sh_mail_msg"));
      }


    /* -- Reset time of last failure. --
     */
    fail_time = 0;

    /* -- Polling, empty queue. --
     */
    if (message == NULL && sh.mailNum.alarm_last == 0)
      SL_RETURN((-1), _("sh_mail_msg"));



    /* ---------  Build complete message. ------------------------ */


    /* ---------- Header  ---------------------------------------- */

    if (mail_subject == NULL)
      {
	sl_strlcpy(mheader, _("Subject: "),            sizeof(mheader));
	sl_strlcat(mheader, sh_unix_time (0),          sizeof(mheader));
	sl_strlcat(mheader, " ",                       sizeof(mheader));
	sl_strlcat(mheader, sh.host.name,              sizeof(mheader));
      }
    else
      {
	sh_mail_get_subject(message, mheader, sizeof(mheader));
      }

    /* RFC 821: Header is terminated by an empty line
     */
    sl_strlcat(mheader, "\015\012\015\012",        sizeof(mheader));

    /* ---------- Message  --------------------------------------- */

    sl_strlcpy(subject, sh_unix_time (0),          sizeof(subject));
    sl_strlcat(subject, " ",                       sizeof(subject));
    sl_strlcat(subject, sh.host.name,              sizeof(subject));
    sl_strlcat(subject, "\r\n",                    sizeof(subject));

#define SH_MAILBUF (4*4096)

    mailMsg     = (char *) SH_ALLOC (SH_MAILBUF);
    bufcompress = (char *) SH_ALLOC (SH_MAILBUF);

    MLOCK(mailMsg     , SH_MAILBUF);
    MLOCK(bufcompress , SH_MAILBUF);

    sl_strlcpy(mailMsg, mheader, SH_MAILBUF);


    sl_strlcat(mailMsg, _("-----BEGIN MESSAGE-----\r\n"), SH_MAILBUF);
    sl_strlcat(mailMsg, subject, SH_MAILBUF);
    sh_util_compress (bufcompress, subject, SH_MAILBUF-KEY_LEN-1);
    if (message != NULL)
      {
	sl_strlcat(mailMsg, message, SH_MAILBUF-(4*KEY_LEN));
	sl_strlcat(mailMsg,  "\r\n", SH_MAILBUF-(4*KEY_LEN));

	sh_util_compress (bufcompress, message, SH_MAILBUF-KEY_LEN-1);
      }

    if (sh.mailNum.alarm_last > 0)
      {
	fifo_temp = SH_ALLOC (sizeof(SH_FIFO));
	fifo_init (fifo_temp);

	while ( NULL != (popMsg = pop_list(fifo_mail)) )
	  {
	    push_list (fifo_temp, popMsg);
	    sl_strlcat(mailMsg, popMsg, SH_MAILBUF-(4*KEY_LEN));
	    sl_strlcat(mailMsg, "\r\n", SH_MAILBUF-(4*KEY_LEN));
	    sh_util_compress (bufcompress, popMsg, SH_MAILBUF-KEY_LEN-1);

	    SH_FREE(popMsg);
	    --sh.mailNum.alarm_last;
	  }
      }

    /* ------ signature block ------------------------------------ */
    
    sl_strlcat(mailMsg, _("-----BEGIN SIGNATURE-----\r\n"), SH_MAILBUF);

    /* Generate new signature key.
     */
    if (isfirst == 1)
      {
	BREAKEXIT(sh_util_keyinit);
	sh_util_keyinit (skey->mailkey_old, KEY_LEN+1);
      }

    /* iterate the key
     */
    sl_strlcpy(skey->mailkey_new,
               sh_tiger_hash (skey->mailkey_old, TIGER_DATA, KEY_LEN),
               KEY_LEN+1);

    if (isfirst == 0)
      {
        /* Sign the message with the signature key.
         */
	theSig = sh_util_siggen (skey->mailkey_new, 
				 bufcompress, sl_strlen(bufcompress));
	sl_strlcat (mailMsg, 
		    theSig,
		    SH_MAILBUF);
      }
    else
      {
        id_audit = time (NULL);

        /* reveal first signature key
         */
        sl_strlcpy(skey->crypt, skey->mailkey_new, KEY_LEN+1);

	BREAKEXIT(sh_util_encode);
        sh_util_encode(skey->crypt, bufcompress, 0, 'A');

        sl_strlcat (mailMsg, skey->crypt, SH_MAILBUF);
        memset (skey->crypt, '\0', KEY_LEN);
        isfirst = 0;
      }
    sl_strlcat (mailMsg, "\r\n", SH_MAILBUF);

    /* X(n) -> X(n-1)
     */
    sl_strlcpy (skey->mailkey_old, skey->mailkey_new, KEY_LEN+1);

    sprintf(subject, _("%06d %010ld::%s\r\n"),         /* known to fit  */
            mailcount, (long) id_audit, sh.host.name);

    sl_strlcat (mailMsg, subject, SH_MAILBUF);
    ++mailcount;

    sl_strlcat (mailMsg, _("-----END MESSAGE-----"), SH_MAILBUF);



    /* ---------- Connect ---------------------------------------- */



    /* -- Catch (ignore) 'broken pipe'.
     */
    new_act.sa_handler = SIG_IGN;
    sigaction (SIGPIPE, &new_act, &old_act);

    i        = 0;
    errcount = 0;

    if (all_in_one == S_FALSE)
      {
	while (address_list[i] != NULL && i < address_num)
	  {
	    connfile = sh_mail_start_conn (i);
	    
	    if (NULL != connfile)
	      {
		status = fwrite (mailMsg, 1, sl_strlen(mailMsg), connfile);
		status -= sl_strlen(mailMsg);
		if (status == 0)
		  status = sh_mail_end_conn (connfile);
	      }
	    if (NULL == connfile ||  status != 0)
	      {
		m_block = 1;
		sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_SRV_FAIL,
				 _("mail"), address_list[i]);
		m_block = 0;
		++errcount;
		if (connfile != NULL) 
		  fclose (connfile);
	      }
	    ++i;
	  }
      }
    else
      {
	connfile = sh_mail_start_conn ( -9 );
	
	if (NULL != connfile)
	  {
	    status = fwrite (mailMsg, 1, sl_strlen(mailMsg), connfile);
	    status -= sl_strlen(mailMsg);
	    if (status == 0)
	      status = sh_mail_end_conn (connfile);
	  }
	if (NULL == connfile ||  status != 0)
	  {
	    m_block = 1;
	    sh_error_handle ((-1), FIL__, __LINE__, 0, MSG_SRV_FAIL,
			     _("mail"), address_list[0]);
	    m_block = 0;
	    errcount = address_num;
	    if (connfile != NULL) 
	      fclose (connfile);
	  }
      }

    
    memset (bufcompress, '\0', SH_MAILBUF);
    MUNLOCK(bufcompress , SH_MAILBUF);
    SH_FREE(bufcompress);

    memset (mailMsg, '\0', SH_MAILBUF);
    MUNLOCK(mailMsg , SH_MAILBUF);
    SH_FREE(mailMsg);

    /* --- Stay responsible for delivery in case of failure --- */

    if (errcount == address_num && fifo_temp != NULL)
      {
        while ( (NULL != (popMsg = pop_list(fifo_temp))) )
          {
            push_list (fifo_mail, popMsg);
	    ++sh.mailNum.alarm_last;
            SH_FREE(popMsg);
          }
	if (message != NULL)
	  {
	    push_list (fifo_mail, message);
	    ++sh.mailNum.alarm_last;
	  }
      }
    else if (fifo_temp != NULL)
      {
        while ( (NULL != (popMsg = pop_list(fifo_temp))) )
          {
            SH_FREE(popMsg);
          }
      }
    if (fifo_temp != NULL)
      SH_FREE(fifo_temp);

    if (connfile != NULL) fclose (connfile);

    /* --- Reset signal. ---
     */
    sigaction (SIGPIPE, &old_act, NULL);

    if (errcount == address_num)
      {
	fail_time = time(NULL);
	SL_RETURN((-1), _("sh_mail_msg"));
      }
    success_time = time(NULL);

    SL_RETURN((0), _("sh_mail_msg"));
}


/*
 *
 * SMTP CODE BELOW
 *
 *
 */

#include <ctype.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

/* missing on HP-UX 10.20 */
#ifndef IPPORT_SMTP
#define IPPORT_SMTP 25
#endif

static int sh_mail_wait(int code, FILE * m_socket);

static char * relay_host = NULL;

int sh_mail_set_relay (char * str_s)
{
  int i = 0;

  SL_ENTER(_("sh_mail_set_relay"));

  if (str_s == NULL)
    SL_RETURN( -1, _("sh_mail_set_relay"));

  if (relay_host != NULL)
    my_free (relay_host);

  i = sl_strlen(str_s) + 1;
  relay_host = sl_malloc (i);
  if (relay_host != NULL)
    sl_strlcpy(relay_host, str_s, i);
  else
    fprintf(stderr, _("ERROR:  sh_mail_set_relay: Out of memory"));
  SL_RETURN( 0, _("sh_mail_set_relay"));
}


/*************************
 *
 * start connection
 * for details on SMTP, see RFC 821 
 */

static time_t time_wait = 300;

static FILE * sh_mail_start_conn (int aFlag)
{
  char       * address;

  int          ecount;

  char         this_address[256];
  char         m_machine[256];
  char         m_user[256];
  char         error_msg[256];
  char         error_call[SH_MINIBUF];
  int          error_num = 0;
  register int i, j, k;
  FILE       * connFile = NULL;
  struct tm  * my_tm;
  time_t       my_time;
  char         my_tbuf[128];

  int                  fd;

  SL_ENTER(_("sh_mail_start_conn"));

  time_wait = 300;

  if (aFlag >= 0)
    address = address_list[aFlag];
  else
    address = address_list[0];

  TPT(( 0, FIL__, __LINE__, _("msg=<aFlag %d address %s>\n"), 
	aFlag, address)); 

  /* -------   split adress ------------------  */

  if (0 == strcmp(address, _("NULL")))
    {
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    } 
    
  if (strchr (address, '@') == NULL) {
    sl_strlcpy(m_user,    address,     256);
    sl_strlcpy(m_machine, _("localhost"), 256);
  } else {
    i = 0;
    while (i < 255 && address[i] != '@') {
      m_user[i] = address[i];
      ++i;
    }

    /* adress[i] = '@' 
     */
    m_user[i] = '\0';
    j = i + 1; k = i; i = 0;
    while (i < 255 && address[i+j] != '\0') {
      m_machine[i] = address[i+j];
      ++i;
    }
    m_machine[i] = '\0';
    if (address[k] != '@' || address[k+i+1] != '\0') 
      {
	SL_RETURN( NULL, _("sh_mail_start_conn"));
      } 
  }


  if (relay_host != NULL) 
    sl_strlcpy (m_machine, relay_host, sizeof(m_machine));
 
  TPT(( 0, FIL__, __LINE__, _("msg=<user %s machine %s>\n"), 
	m_user, m_machine)); 

  fd = connect_port (m_machine, IPPORT_SMTP, 
		     error_call, &error_num, error_msg, 256);

  if (fd < 0)
    {
      sh_error_handle((-1), FIL__, __LINE__, error_num, MSG_TCP_NETRP, 
		      error_msg, (long) IPPORT_SMTP, error_call);
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    }

  /* associate a FILE structure with it
   */
  MBLK( connFile = fdopen (fd, "r+"); )
  if (connFile == NULL) 
    {
      TPT(( 0, FIL__, __LINE__, _("msg=<fdopen() failed>\n")));
      close(fd);
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    }


  /* say HELO to the other socket
   */
  if (!sh_mail_wait (220, connFile)) 
    {
      TPT(( 0, FIL__, __LINE__, _("msg=<Timeout>\n")));
      MBLK( fclose(connFile); )
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    }

  fflush(connFile);

  if (is_numeric(sh.host.name))
    {
      TPT(( 0, FIL__, __LINE__, _("msg=<HELO [%s]>%c%c"), 
	    sh.host.name, 13, 10));
    }
  else
    {
      TPT(( 0, FIL__, __LINE__, _("msg=<HELO %s>%c%c"), 
	    sh.host.name, 13, 10));
    }
  if (is_numeric(sh.host.name))
    fprintf(connFile, _("HELO [%s]%c%c"), sh.host.name, 13, 10);
  else
    fprintf(connFile, _("HELO %s%c%c"), sh.host.name, 13, 10);

  fflush(connFile);

  if (!sh_mail_wait(250, connFile)) 
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_NET, 
		      _("HELO failed"), _("sh_mail_start_conn"), 
		      _("mail"), sh.host.name);

      TPT(( 0, FIL__, __LINE__, _("msg=<Timeout.>\n")));
      MBLK( fclose(connFile); )
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    }

  /* tell them who we are
   */
  sl_strlcpy (this_address, DEFAULT_SENDER, 256);
  sl_strlcat (this_address, "@", 256);
  sl_strlcat (this_address, sh.host.name, 256);

  TPT(( 0, FIL__, __LINE__,  _("msg=<MAIL FROM:<%s>>%c%c"), 
	this_address, 13, 10));

  fflush(connFile);
  fprintf(connFile, _("MAIL FROM:<%s>%c%c"), this_address, 13, 10);
  fflush(connFile);

  if (!sh_mail_wait(250, connFile)) 
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_NET, 
		      _("MAIL FROM failed"), _("sh_mail_start_conn"), 
		      _("mail"), this_address);
      TPT(( 0, FIL__, __LINE__, _("msg=<Timeout.>\n")));
      MBLK( fclose(connFile); )
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    }

  /* tell them who to send mail to
   */
  if (aFlag >= 0)
    {
      TPT(( 0, FIL__, __LINE__,  _("msg=<RCPT TO:<%s>>%c%c"), 
	    address, 13, 10)); 

      fflush(connFile);
      fprintf(connFile, _("RCPT TO:<%s>%c%c"), address, 13, 10); 
      fflush(connFile);

      if (!sh_mail_wait(250, connFile)) 
	{
	  sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_NET, 
			  _("RCPT TO failed"), _("sh_mail_start_conn"), 
			  _("mail"), address);
	  TPT(( 0, FIL__, __LINE__, _("msg=<Timeout.>\n")));
	  MBLK( fclose(connFile); )
	  SL_RETURN( NULL, _("sh_mail_start_conn"));
	}
    }
  else
    {
      ecount = 0;
      for (i = 0; i < address_num; ++i)
	{
	  if (address_list[i] == NULL)  /* paranoia */
	    break;
	  TPT(( 0, FIL__, __LINE__,  _("msg=<RCPT TO:<%s>>%c%c"), 
		address_list[i], 13, 10)); 
	  
	  fflush(connFile);
	  fprintf(connFile, _("RCPT TO:<%s>%c%c"), address_list[i], 13, 10); 
	  fflush(connFile);
	  
	  if (!sh_mail_wait(250, connFile)) 
	    {
	      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_NET, 
			      _("RCPT TO failed"), _("sh_mail_start_conn"), 
			      _("mail"), address_list[i]);

	      TPT(( 0, FIL__, __LINE__, _("msg=<Timeout.>\n")));
	      ++ecount;
	    }
	}
      if (ecount == address_num)
	{
	  MBLK( fclose(connFile); )
	  SL_RETURN( NULL, _("sh_mail_start_conn"));
	}
    }

  /* Send the message 
   */
  TPT(( 0, FIL__, __LINE__,  _("msg=<DATA>%c%c"), 13, 10)); 

  fflush(connFile);
  fprintf(connFile, _("DATA%c%c"), 13, 10);      
  fflush(connFile);

  if (!sh_mail_wait(354, connFile)) 
    {
      sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_NET, 
		      _("DATA failed"), _("sh_mail_start_conn"), 
		      _("mail"), address);
      TPT(( 0, FIL__, __LINE__, _("msg=<Timeout.>\n")));
      MBLK( fclose(connFile); )
      SL_RETURN( NULL, _("sh_mail_start_conn"));
    }


  my_time = time(NULL);
  my_tm   = localtime(&my_time);
  (void)    strftime(my_tbuf, 127, _("%a, %d %b %Y %H:%M:%S %Z"), my_tm);

  TPT(( 0, FIL__, __LINE__,  _("msg=<From: <%s>%c%cTo: <%s>%c%cDate: %s>%c%c"),
	this_address, 13, 10, address, 13, 10, my_tbuf, 13, 10));

  fflush(connFile);
  fprintf(connFile,
	  _("From: <%s>%c%c"\
	    "To: <%s>%c%c"\
	    "Date: %s%c%c"),
	  this_address, 13, 10,
	  address, 13, 10,
	  my_tbuf, 13, 10);

  SL_RETURN( connFile, _("sh_mail_start_conn"));
}

/*************************
 *
 * end connection
 *
 */

static int sh_mail_end_conn (FILE * connFile)
{
  SL_ENTER(_("sh_mail_end_conn"));

  time_wait = 300;

  fflush(connFile);
  fprintf(connFile, _("%c%c.%c%c"), 13, 10, 13, 10);   
  fflush(connFile);

  TPT(( 0, FIL__, __LINE__, _("msg=<message end written>\n")));

  if (sh_mail_wait(250, connFile))
    {  
      fflush(connFile);
      fprintf(connFile, _("QUIT%c%c"), 13, 10);
      TPT(( 0, FIL__, __LINE__, _("msg=<exit>\n")));

      SL_RETURN (0, _("sh_mail_end_conn"));
    }
    
  sh_error_handle((-1), FIL__, __LINE__, 0, MSG_E_NET, 
		  _("QUIT failed"), _("sh_mail_end_conn"), 
		  _("mail"), _("SMTP server"));

  TPT(( 0, FIL__, __LINE__, _("msg=<abnormal exit>\n")));

  SL_RETURN ((-1), _("sh_mail_end_conn"));
}



/****************************
 *
 * Handle server replies
 *
 *
 */

static jmp_buf wait_timeout;

static void sh_mail_alarmhandle (int mysignal)
{
  if (mysignal == SIGALRM)   /* use 'signal' to avoid compiler warning */
    {  
      alarm(0);
      longjmp(wait_timeout, 1);
    }
}

static int sh_mail_wait(int code, FILE * m_socket)
{
  int rcode, g;

  enum { 
    WAIT_CODE_START, 
    WAIT_CODE, 
    WAIT_NL, 
    WAIT_NL_CONT 
  } state;

  time_t waited_time;

  struct  sigaction  def_act;
  struct  sigaction  new_act;
  sigset_t           unblock;

  sigemptyset(&unblock);
  sigaddset  (&unblock, SIGALRM);

  def_act.sa_handler = SIG_IGN;

  new_act.sa_handler = sh_mail_alarmhandle;
  sigemptyset( &new_act.sa_mask );       /* set an empty mask       */
  new_act.sa_flags = 0;                  /* init sa_flags           */

  SL_ENTER(_("mail_wait"));
  
  /* alarm was triggered
   */
  if (setjmp(wait_timeout) != 0)
    {
      alarm(0);
      sigaction (SIGALRM, &def_act, NULL);
      sigprocmask(SIG_UNBLOCK, &unblock, NULL);
      TPT((0, FIL__, __LINE__, _("msg=<mail_wait: timeout>\n"))); 
      SL_RETURN( 0, _("mail_wait"));
    }

  waited_time = time(NULL);

  /* timeout after 5 minutes
   */
  alarm(0);
  sigaction (SIGALRM, &new_act, &def_act);
  alarm(time_wait);

  rcode = 0;
  state = WAIT_CODE_START;

  while (!feof(m_socket) && !ferror(m_socket)) {

    if ( (g=fgetc(m_socket)) == EOF)
      {
	alarm(0);
	TPT((0, FIL__, __LINE__, _("msg=<mail_wait: EOF>\n"))); 
	sigaction (SIGALRM, &def_act, NULL); 
	SL_RETURN( 0, _("mail_wait")); 
      }

    switch(state) {

      /* wait for start of a numerical code
       */
    case WAIT_CODE_START:
      if (isspace(g))
	break;             /* Skip white space                    */
      if (!isdigit(g))
	return 0;          /* No leading number                   */
      rcode = g-'0';       /* convert to number                   */
      state = WAIT_CODE;
      break;
      
      /* wait for completion of numerical code
       */
    case WAIT_CODE:
      if (isdigit(g)) {
	rcode = rcode * 10 + (g-'0'); /* next digit               */
	break;
      }
      state = ((g == '-') ?  WAIT_NL_CONT :  WAIT_NL); 
      break;
      
      /* wait for newline, then return with status code
       */
    case WAIT_NL:
      if (g != '\n')
	break;
      alarm(0);                            /* Disable alarm       */
      sigaction (SIGALRM, &def_act, NULL); /* Reset handler       */
      TPT((0, FIL__, __LINE__, 
	   _("msg=<mail_wait: OK got %d (%d) need %d (%d)>\n"),
	   rcode, (int)(rcode/100), code, (int)(code/100) ));
      g = ((int)(rcode/100) == (int)(code/100)) ? 1 : 0;
      waited_time = time(NULL) - waited_time;
      time_wait -= waited_time;
      TPT((0, FIL__, __LINE__, 
	   _("msg=<mail_wait: time_wait reduced to %d sec>\n"),
	   (int) time_wait));
      SL_RETURN( (g), _("mail_wait")) ;
      
      /* wait for continuation line
       */
    case WAIT_NL_CONT:
      if (g == '\n')
	state = WAIT_CODE_START;  /* There is a continuation line */
      break; 
      
    default:
      alarm(0);                            /* Disable alarm       */
      sigaction (SIGALRM, &def_act, NULL); /* Reset handler       */
      TPT((0, FIL__, __LINE__, _("msg=<mail_wait: bad>\n"))); 
      SL_RETURN( 0, _("mail_wait")); 
      
    }
  }

  alarm(0);                            /* Disable alarm       */
  sigaction (SIGALRM, &def_act, NULL); /* Reset handler       */

  TPT((0, FIL__, __LINE__, _("msg=<mail_wait: failed>\n"))); 

  /* Failed, EOF or error on socket */
  SL_RETURN( 0, _("mail_wait")); 
}

  

/* if defined(SH_WITH_MAIL) */
#endif



