/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2001 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#define SH_SYSCALL_CODE

#include "config_xor.h"

#ifdef SH_USE_KERN
#ifndef HAVE_LINUX_MODULE_H
#error missing or unusable include file linux/module.h
#endif
#ifndef HAVE_LINUX_UNISTD_H
#error missing or unusable include file linux/module.h
#endif

#define __KERNEL__
#include <linux/module.h>
#undef __KERNEL__
#include <linux/unistd.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>
#include <sys/wait.h>
#include <signal.h>



#ifdef SH_USE_KERN

#undef  FIL__
#define FIL__  _("sh_kern.c")

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif


#include "samhain.h"
#include "sh_utils.h"
#include "sh_error.h"
#include "sh_modules.h"
#include "sh_kern.h"
#include "sh_ks_xor.h"



sh_rconf sh_kern_table[] = {
  {
    N_("severitykernel"),
    sh_kern_set_severity
  },
  {
    N_("kernelcheckactive"),
    sh_kern_set_activate
  },
  {
    N_("kernelcheckinterval"),
    sh_kern_set_timer
  },
  {
    NULL,
    NULL
  },
};


static time_t  lastcheck;
static int     ShKernActive   = S_TRUE;
static int     ShKernInterval = 300;
static int     ShKernSeverity = SH_ERR_SEVERE;
static int     ShKernDelay    = 100; /* milliseconds */




int query_module(const char *name, int which, void *buf, size_t bufsize,
                 size_t *ret);

struct new_module_symbol
{
  unsigned long value;
  unsigned long name;
};

#define QM_SYMBOLS     4


long my_address (char *sym_name)
{
  struct new_module_symbol *buf;
  size_t ret, j, size;
  long   retval;
  int    status;
  char   *p;

  SL_ENTER(_("my_address"));

  buf = (struct new_module_symbol *) 
    sl_malloc (sizeof(struct new_module_symbol));

  if (buf == NULL)
    {
      TPT((0, FIL__, __LINE__, _("msg=<Out of memory>\n")));
      SL_RETURN( -1, _("my_address"));
    }

  size = sizeof(struct new_module_symbol);

  while (0 != query_module(NULL, QM_SYMBOLS, buf, 
			   size,
			   &ret))
    {
      if (errno == ENOSPC)
	{
	  my_free(buf);
	  size = ret;
	  buf  = (struct new_module_symbol *) sl_malloc (size);
	  if (buf == NULL)
	    {
	      TPT((0, FIL__, __LINE__, _("msg=<Out of memory>\n")));
	      SL_RETURN( -1, _("my_address"));
	    }
	}
      else
	{
	  status = errno;
	  TPT((0, FIL__, __LINE__, _("msg=<query_module: %s>\n"), 
	       sh_error_message(status)));
	  my_free(buf);
	  SL_RETURN( -1, _("my_address"));
	}
    }

  
  for (j = 0; j < ret; ++j)
    {
      p = (char *) buf; p += buf[j].name;
      if(strstr(p, sym_name))
	{
	  retval = (long) buf[j].value;
	  my_free(buf);
	  SL_RETURN( retval, _("my_address"));
	}
    }

  TPT((0, FIL__, __LINE__, _("msg=<%s kmem offset not found>\n"), sym_name));
  my_free(buf);
  SL_RETURN( -1, _("my_address"));
}


int sh_kern_check_internal ()
{
  int kd;
  int res;
  pid_t mpid;
  int mpipe[2];
  int i, j, status = 0;
  unsigned int kaddr;
  unsigned long kmem_call_table[512];

#ifdef SH_SYSCALL_CODE
  unsigned int kmem_code_table[512][2];
#endif

  unsigned char new_system_call_code[256];

#ifdef SH_USE_LKM
  static int check_getdents      = 0;
#ifdef __NR_getdents64
  static int check_getdents64    = 0;
#endif
#endif

  SL_ENTER(_("sh_kern_check_internal"));

  kaddr = my_address (_("sys_call_table"));

  if (kaddr == (unsigned int) -1)
    {
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_SUBGEN,
		       _("error finding sys_call_table"),
		       _("kern_check_internal") );
      SL_RETURN( (-1), _("sh_kern_check_internal"));
    }
  
  kd = aud_open(FIL__, __LINE__, SL_YESPRIV, _("/dev/kmem"), O_RDONLY, 0);
  
  if (kd < 0)
    {
      status = errno;
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_SUBGEN,
		       _("error opening /dev/kmem"),
		       _("kern_check_internal") );
      SL_RETURN( (-1), _("sh_kern_check_internal"));
    }

  status = aud_pipe(FIL__, __LINE__, mpipe);

  if (status == 0)
    {
      mpid = aud_fork(FIL__, __LINE__);

      switch (mpid) 
	{
	case -1:
	  status = -1;
	  break;
	case 0:                       /* child */
	  status = close(mpipe[0]);
	  setpgid(0, 0);
	  
	  if(status == 0)
	    {
	      if(lseek(kd, (off_t)kaddr, SEEK_SET) == -1)
		{
		  status = -2;
		}
	    }
	  
	  if(status == 0)
	    {
	      usleep (ShKernDelay * 1000); /* milliseconds */
	      if (sizeof(kmem_call_table) != 
		  read(kd, &kmem_call_table, sizeof(kmem_call_table)))
		{
		  status = -3;
		}
	    }

#ifdef SH_SYSCALL_CODE
	  if(status == 0)
	    {
	      memset(kmem_code_table, 0, sizeof(kmem_code_table));
	      for (j = 0; j < SH_MAXCALLS; ++j) 
		{
		  if (sh_syscalls[j].name == NULL || 
		      sh_syscalls[j].addr == 0UL)
		    break;

		  lseek (kd, sh_syscalls[j].addr, SEEK_SET);
		  read  (kd, &(kmem_code_table[j][0]), 
			 2 * sizeof(unsigned int));
		}
	    }
#endif

	  if(status == 0)
	    {
	      lseek (kd, system_call_addr, SEEK_SET);
	      read  (kd, new_system_call_code, 256);
	    }

	  if(status == 0)
	    {
	      status = 
		write(mpipe[1], &kmem_call_table, sizeof(kmem_call_table));
#ifdef SH_SYSCALL_CODE
	      if(status > 0)
		{
		  status = 
		    write(mpipe[1], &kmem_code_table, sizeof(kmem_code_table));
		}
#endif
	      if(status > 0)
		{
		  status = 
		    write(mpipe[1], new_system_call_code, 256);
		}
	    }
	  _exit( (status >= 0) ? 0 : status);
	  break;
	  
	default:
	  close (mpipe[1]);
	  close (kd);
	  usleep (ShKernDelay * 1000); /* milliseconds */
	  if (sizeof(kmem_call_table) != 
	      read(mpipe[0], &kmem_call_table, sizeof(kmem_call_table)))
	    status = -4;
	  else
	    status = 0;
#ifdef SH_SYSCALL_CODE
	  if(status == 0)
	    {
	      if (sizeof(kmem_code_table) != 
		  read(mpipe[0], &kmem_code_table, sizeof(kmem_code_table)))
		status = -5;
	      else
		status = 0;
	    }
#endif	  
	  if(status == 0)
	    {
	      if (256 != read(mpipe[0], new_system_call_code, 256))
		status = -6;
	      else
		status = 0;
	    }

	  if (status < 0)
	    res = waitpid(mpid, NULL,    WNOHANG|WUNTRACED);
	  else 
	    {
	      res = waitpid(mpid, &status, WNOHANG|WUNTRACED);
	      if (res == 0 && 0 != WIFEXITED(status))
		status = WEXITSTATUS(status);
	    }
	  close (mpipe[0]);
	  if (res <= 0)
	    {
	      kill(mpid, 9);
	      waitpid(mpid, NULL, 0);
	    }
	  break;
	}
    }

  if ( status < 0)
    {
      sh_error_handle ((-1), FIL__, __LINE__, status, MSG_E_SUBGEN,
		       _("error reading from /dev/kmem"),
		       _("kern_check_internal") );
      SL_RETURN( (-1), _("sh_kern_check_internal"));
    }

  for (i = 0; i < (SYS_CALL_LOC + 4); ++i) 
    {
      if (system_call_code[i] != new_system_call_code[i])
	{
	  sh_error_handle (ShKernSeverity, FIL__, __LINE__, 
			   status, MSG_KERN_POL_CO,
			   new_system_call_code[i], 0,
			   system_call_code[i], 0,
			   _("system_call (int 80h vector)")
			   );
	  for (j = 0; j < (SYS_CALL_LOC + 4); ++j)
	    system_call_code[j] = new_system_call_code[j];
	  break;
	}
    }
  
  for (i = 0; i < SH_MAXCALLS; ++i) 
    {
      if (sh_syscalls[i].name == NULL || sh_syscalls[i].addr == 0UL)
	break;

      /*
      fprintf(stderr, "Kernel: %#lx Map: %#lx  %s\n", 
	      kmem_call_table[i], 
	      sh_syscalls[i].addr, 
	      sh_syscalls[i].name);
      */
      
#ifdef SH_USE_LKM
      if (sh_syscalls[i].addr != kmem_call_table[i])
	{
	  if (check_getdents == 0 && 
	      0 == strcmp(_(sh_syscalls[i].name), _("sys_getdents")))
	    {
	      check_getdents = 1;
	      sh_error_handle (SH_ERR_WARN, FIL__, __LINE__, 
			       status, MSG_E_SUBGEN,
			       _("Modified kernel syscall (expected)."),
			       _(sh_syscalls[i].name) );
	    }
#ifdef __NR_getdents64
	  else if  (check_getdents64 == 0 && 
	      0 == strcmp(_(sh_syscalls[i].name), _("sys_getdents64")))
	    {
	      check_getdents64 = 1;
	      sh_error_handle (SH_ERR_WARN, FIL__, __LINE__, 
			       status, MSG_E_SUBGEN,
			       _("Modified kernel syscall (expected)."),
			       _(sh_syscalls[i].name) );
	    }
#endif
	  else
	    {
	      sh_error_handle (ShKernSeverity, FIL__, __LINE__, 
			       status, MSG_KERN_POLICY,
			       kmem_call_table[i],
			       sh_syscalls[i].addr,
			       _(sh_syscalls[i].name)
			       );
	    }
	  sh_syscalls[i].addr = kmem_call_table[i];
	}
#else
      if (sh_syscalls[i].addr != kmem_call_table[i])
	{
	  sh_error_handle (ShKernSeverity, FIL__, __LINE__, 
			   status, MSG_KERN_POLICY,
			   kmem_call_table[i],
			   sh_syscalls[i].addr,
			   _(sh_syscalls[i].name)
			   );
	  sh_syscalls[i].addr = kmem_call_table[i];
	}
#endif
#ifdef SH_SYSCALL_CODE
      if (sh_syscalls[i].code[0] != kmem_code_table[i][0] || 
	  sh_syscalls[i].code[1] != kmem_code_table[i][1])
	{
	  sh_error_handle (ShKernSeverity, FIL__, __LINE__, 
			   status, MSG_KERN_POL_CO,
			   kmem_code_table[i][0],  kmem_code_table[i][1],
			   sh_syscalls[i].code[0], sh_syscalls[i].code[1],
			   _(sh_syscalls[i].name)
			   );
	  sh_syscalls[i].code[0] = kmem_code_table[i][0];
	  sh_syscalls[i].code[1] = kmem_code_table[i][1];
	}
#endif
    }

  SL_RETURN( (0), _("sh_kern_check_internal"));
}

/*************
 *
 * module init
 *
 *************/
int sh_kern_init ()
{
  SL_ENTER(_("sh_kern_init"));
  if (ShKernActive == S_FALSE)
    SL_RETURN( (-1), _("sh_kern_init"));

  lastcheck  = time (NULL);
  sh_kern_check_internal ();
  SL_RETURN( (0), _("sh_kern_init"));
}

/*************
 *
 * module cleanup
 *
 *************/
int sh_kern_end ()
{
  return (0);
}


/*************
 *
 * module timer
 *
 *************/
int sh_kern_timer (unsigned long tcurrent)
{
  if ((time_t) (tcurrent - lastcheck) >= ShKernInterval)
    {
      lastcheck  = tcurrent;
      return (-1);
    }
  return 0;
}

/*************
 *
 * module check
 *
 *************/
int sh_kern_check ()
{
  sh_error_handle (SH_ERR_INFO, FIL__, __LINE__, EINVAL, MSG_E_SUBGEN,
		   _("Checking kernel syscalls"),
		   _("kern_check") );
  return (sh_kern_check_internal ());
}

/*************
 *
 * module setup
 *
 *************/

int sh_kern_set_severity  (char * c)
{
  char tmp[32];
  tmp[0] = '='; tmp[1] = '\0';
  sl_strlcat (tmp, c, 32);
  sh_error_set_level (tmp, &ShKernSeverity);
  return 0;
}

int sh_kern_set_timer (char * c)
{
  long val;

  SL_ENTER(_("sh_kern_set_timer"));

  val = strtol (c, (char **)NULL, 10);
  if (val <= 0)
    sh_error_handle ((-1), FIL__, __LINE__, EINVAL, MSG_EINVALS,
                      _("kern timer"), c);

  val = (val <= 0 ? 60 : val);

  ShKernInterval = (time_t) val;
  SL_RETURN( 0, _("sh_kern_set_timer"));
}

int sh_kern_set_activate (char * c)
{
  int i;
  SL_ENTER(_("sh_kern_set_activate"));
  i = sh_util_flagval(c, &ShKernActive);
  SL_RETURN(i, _("sh_kern_set_activate"));
}

#endif

/* #ifdef SH_USE_UTMP */
#endif



