/* SAMHAIN file system integrity testing                                   */
/* Copyright (C) 2002 Rainer Wichmann                                      */
/*                                                                         */
/*  This program is free software; you can redistribute it                 */
/*  and/or modify                                                          */
/*  it under the terms of the GNU General Public License as                */
/*  published by                                                           */
/*  the Free Software Foundation; either version 2 of the License, or      */
/*  (at your option) any later version.                                    */
/*                                                                         */
/*  This program is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of         */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          */
/*  GNU General Public License for more details.                           */
/*                                                                         */
/*  You should have received a copy of the GNU General Public License      */
/*  along with this program; if not, write to the Free Software            */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <errno.h>
#include <sys/types.h>
#include <unistd.h>

/* 
   gcc -Wall -O2 -o mysched sh_schedule.c -DTESTONLY
 */
#ifndef TESTONLY

#include "config_xor.h"

#undef  FIL__
#define FIL__  _("sh_schedule.c")

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
#define SCHEDULER_YES
#endif

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#include "sh_mem.h"

/* TESTONLY */
#else

#define SCHEDULER_YES
#include <time.h>

#endif

#include "sh_schedule.h"



#ifdef SCHEDULER_YES

/************************************************
 * 
 * Scheduler class - private area
 *
 ************************************************/


static const int  sh_schedule_max[5] = { 59, 23, 31, 12, 7 };
static const int  sh_schedule_min[5] = {  0,  0,  0,  0, 0 };

int test_val (int i, int min, int max, int min_step, 
	      time_t * last, time_t now, int nval)
{
  /* don't miss a minute's task
   * IDEA:  set last = now after first check (? seems to work)
   */
  if (i == 0 && max == min && nval > max 
      /* && ( ((now - *last) > min_step) || (*last == (time_t)-1) ) */ )
    {
      if (*last == (time_t)-1)
	{
	  /* fake execution at nval-max
	   */
	  *last = now - 60 * (nval-max);
	  return 0;
	}
      if ((now - *last) > min_step)
	return 1;
    }

  /* out of range
   */
  if (nval > max || nval < min) 
    return 0;

  /* first call - invalid last_exec
   */
  if (*last == (time_t)-1)
    return 1;

  /* before min_step - too early (e.g. same minute)
   */
  if (  (now - *last) <= min_step)
    return 0;

  return 1;
}

int test_sched (sh_schedule_t * isched)
{
  time_t now;
  struct tm * tval;
  int count, i, nval;

  if (!isched)
    return 0;

  now  = time(NULL);
  tval = localtime(&now);

  count = 0;
  for (i = 0; i < 5; ++i)
    {
      if      (i == 0) nval = tval->tm_min;
      else if (i == 1) nval = tval->tm_hour;
      else if (i == 2) nval = tval->tm_mday;
      else if (i == 3) nval = tval->tm_mon;
      else             nval = tval->tm_wday;
      count += test_val (i, isched->min[i], isched->max[i], 
			 isched->min_step, &(isched->last_exec), 
			 now, nval);
    }
  if (count == 5)
    {
      isched->last_exec = now;
      return 1;
    }

  return 0;
}

char DayNames[7][3] = { "sun", "mon", "tue", "wed", "thu", "fri", "sat" };
char MonNames[12][3] = { "jan", "feb", "mar", "apr", "may", "jun", 
		       "jul", "aug", "sep", "oct", "nov", "dec" };


int parse_func (int i, char * p)
{
  int j, k, l;
  char *tail;

  errno = 0;
  j = strtol(p, &tail, 10);

  if (errno)     /* overflow          */
    return -1;
  if (tail != p) /* numeric           */
    return j;
  if (i < 3)     /* names not allowed */
    return -1;

  if (i == 3)
    {
      for (j = 0; j < 12; ++j) {
	l = 0;
	for (k = 0; k < 3; ++k)
	  if (p[k] != '\0' && tolower(p[k]) == MonNames[j][k]) ++l;
	if (l == 3)
	  return j;
      }
    }
  if (i == 4)
    {
      for (j = 0; j < 7; ++j) {
	l = 0;
	for (k = 0; k < 3; ++k)
	  if (p[k] != '\0' && tolower(p[k]) == DayNames[j][k]) ++l;
	if (l == 3)
	  return j;
      }
    }

  return -1;
}  

int parse_token(int i, sh_schedule_t * isched, char * p)
{
  char * q;

  if ( NULL != (q = strchr(p, ',')))
    return -1;

  if (*p == '*')
    {
      isched->min[i] = sh_schedule_min[i];
      isched->max[i] = sh_schedule_max[i];
    }
  else 
    {
      isched->min[i] = parse_func(i, p);
      if (i == 4 && isched->min[i] == 7)
	isched->min[i] = 0;
      if (isched->min[i] < sh_schedule_min[i] || 
	  isched->min[i] > sh_schedule_max[i])
	{
	  return -1;
	}
      if ( NULL != (q = strchr(p, '-')))
	{
	  ++q;
	  isched->max[i] = parse_func(i, q);
	  if (i == 4 && isched->max[i] == 7)
	    isched->max[i] = 0;
	  if (isched->max[i] < sh_schedule_min[i] || 
	      isched->max[i] > sh_schedule_max[i] ||
	      isched->max[i] < isched->min[i])
	    {
	      return -1;
	    }
	}
      else
	isched->max[i] = isched->min[i];
    }

  if ( NULL != (q = strchr(p, '/')))
    {
      ++q;
      isched->step[i] = atoi(q);
      if (isched->step[i] < 1 || isched->step[i] > sh_schedule_max[i])
	{
	  return -1;
	}
      if (i == 4 && isched->step[i] == 7)
	isched->step[i] = 6;
    }
  else
    {
      isched->step[i] = 1;
    }

  switch (i) 
    {
    case 0:
      if (isched->max[i] == isched->min[i])
	isched->min_step = 3599;
      else
	isched->min_step = (isched->step[i] * 60) - 1;
      break;
    case 1:
      if (isched->max[i] == isched->min[i])
	{
	  if (isched->min_step == 3599)
	    isched->min_step = 86399;
	}
      else
	{
	  if (isched->min_step == 3599)
	    isched->min_step = (isched->step[i] * 3600) - 1;
	}
      break;
    default:
      break;
    }
     
  return 0;
}

int parse_sched (const char * ssched, sh_schedule_t * isched)
{
  char * p;
  char * copy;
  int    i = 0;

  if (!ssched || !isched)
    return -1;

#ifdef TESTONLY
  copy = malloc(strlen(ssched)+1);
#else
  copy = SH_ALLOC(strlen(ssched)+1);
#endif
  strcpy(copy, ssched);                            /* known to fit  */

  p = strtok(copy, " \t");
  if (!p)
    goto err; 
  if (parse_token(i, isched, p) == -1)
    goto err;

  for (i = 1; i < 5; ++i)
    {
      p = strtok(NULL, " \t");
      if (!p)
	goto err; 
      if (parse_token(i, isched, p) == -1)
	goto err;
    }

  /*
  for (i = 0; i < 5; ++i)
    printf("%2d MIN  %3d  MAX  %3d  STEP  %3d\n", 
	   i, isched->min[i], isched->max[i], isched->step[i]);
  */

  isched->last_exec = (time_t)-1;

  SH_FREE(copy);
  return 0;

 err:
  SH_FREE(copy);
  return -1;
}

int create_sched (const char * ssched, sh_schedule_t * isched)
{
  int j;

  if (!isched || !ssched)
    return -1;

  j = parse_sched(ssched, isched);

#ifdef TESTONLY
  if (j == 0)
    {
      int i;
      for (i = 0; i < 5; ++i)
	printf("%2d MIN  %3d  MAX  %3d  STEP  %3d\n", 
	       i, isched->max[i], isched->min[i], isched->step[i]);
      printf("MINSTEP  %3d\n", isched->min_step);
    }
#endif

  return j;
}

/* #ifdef SCHEDULER_YES */
#endif

/**************************************************
 *
 * Schedule class - Test driver
 *
 **************************************************/
#ifdef TESTONLY

int main(int argc, char * argv[])
{
  sh_schedule_t isched;

  if (argc < 2)
    {
      fprintf(stderr, "Usage: %s 'schedule'\n", argv[0]);
      exit (1);
    }

  if (create_sched(argv[1], &isched) < 0)
    {
      fprintf(stderr, "Bad schedule <%s>\n", argv[1]);
      exit (1);
    }

  while (1 == 1)
    {
      if (test_sched(&isched))
	printf("EXECUTE at: %s", ctime(&(isched.last_exec)));
      sleep (1);
    }
  return 0;
}
#endif
