/* --------------------------------------------------------------
 * 
 * The developement of this library has been stimulated by reading 
 * a paper on 'Robust Programming' by Matt Bishop, although
 * not all of his ideas might be implemented in the same 
 * strictness as discussed in the paper.
 *
 * --------------------------------------------------------------
 */

#ifndef SL_SLIB_H
#define SL_SLIB_H

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>

#include "config_xor.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

/****************

		 -- Defined in config.h. --

		 #ifndef _(string)
		 #define _(string) string
		 #endif
		 
		 #ifndef N_(string)
		 #define N_(string) string
		 #endif

*****************/

/* --------------------------------------------------------------
 * 
 * Typedefs, global variables, macros.
 *
 * --------------------------------------------------------------
 */

extern  long int sl_errno;              /* Global error variable.         */


/* The ticketing system; used to hide internals from the
 * programmer.
 */
typedef long int SL_TICKET;             /* Unique ID for opened files.    */


/*
 * TRUE, FALSE
 */
#define SL_TRUE  1
#define SL_FALSE 0



/*
 *  The following macros are provided:
 *  
 *  SL_ISERROR(x)       TRUE if return status of 'x' is an error code.
 *  SL_REQUIRE(x, xstr) Abort if 'x' is false.
 *  SL_ENTER(s)         Trace entry in    function 's'.
 *  SL_RETURN(x, s)     Trace return from function 's'.
 */


/*
 * The error codes.
 */
#define SL_ENONE         0

#define SL_ENULL     -1024     /* Invalid use of NULL pointer.         */
#define SL_ERANGE    -1025     /* Argument out of range.               */
#define SL_ETRUNC    -1026     /* Result truncated.                    */
#define SL_EREPEAT   -1027     /* Illegal repeated use of function.    */

#define SL_EINTERNAL -1028     /* Internal error.                      */
#define SL_ETICKET   -1029     /* Bad ticket.                          */
#define SL_EBADFILE  -1030     /* File access error. Check errno.      */
#define SL_EBOGUS    -1031     /* Bogus file.                          */
#define SL_EMEM      -1032     /* Out of memory.                       */
#define SL_EUNLINK   -1033     /* Unlink error. Check errno.           */
#define SL_EREWIND   -1034     /* Rewind error. Check errno.           */
#define SL_EFORWARD  -1035     /* Forward error. Check errno.          */
#define SL_EREAD     -1036     /* Read error. Check errno.             */
#define SL_EWRITE    -1037     /* Write error. Check errno.            */
#define SL_ESYNC     -1038     /* Write error. Check errno.            */

#define SL_EBADNAME  -1040     /* Invalid name.                        */
#define SL_ESTAT     -1041     /* stat of file failed. Check errno.    */

#define SL_EBADUID   -1050	/* Owner not trustworthy.              */
#define SL_EBADGID   -1051	/* Group writeable and not trustworthy.*/
#define SL_EBADOTH   -1052	/* World writeable.                    */



/*
 * All int functions return SL_NONE on success.
 */

#ifdef  __cplusplus
extern "C" {
#endif

  int dlog (int flag, char * file, int line, const char *fmt, ...);

  char * sl_get_errmsg();

  void   my_free   (void * ptr);

  void * sl_malloc (size_t size);

  /* ---------------------------------------------------------------- 
   *
   *    String handling routines
   *
   * ---------------------------------------------------------------- */

  /* 
   * Copy src to dst. siz is the length of dst.
   */
  int sl_strlcpy(char * dst, const char * src, size_t siz);

  /* 
   * Append src to dst. siz is the length of dst.
   */
  int sl_strlcat(char * dst, const char *src,  size_t siz);

  /*
   * An implementation of vsnprintf. va_start/va_end are in the caller
   * function.
   */
  int sl_vsnprintf(char *str, size_t n,
		   const char *format, va_list vl );

  /*
   * An implementation of snprintf.
   */
  int sl_snprintf(char *str, size_t n,
		  const char *format, ... );
  
  /*
   * A robust drop-in replacement of strncpy. strlcpy is preferable.
   */
  char * sl_strncpy(char *dst, const char *src, size_t size);

  /*
   * Robust strncat.
   */
  char * sl_strncat(char *dst, const char *src, size_t n);

  /*
   * strstr
   */
  char * sl_strstr (const char * haystack, const char * needle); 

  /*
   * robust strncmp replacement
   */
  int sl_strncmp(char * a, char * b, size_t n);

  /*
   * robust strcmp replacement
   */
  int sl_strcmp(char * a, char * b);

  /*
   * robust strlen replacement
   */
#if 0
  size_t sl_strlen(const char *s);
#else
#define sl_strlen(arg) ((arg == NULL) ? 0 : (strlen(arg)))
#endif

  /* ---------------------------------------------------------------- 
   *
   *    Privilege handling routines
   *
   * ---------------------------------------------------------------- */

  /*
   * ONE OF THE FOLLOWING THREE FUNCTIONS
   * SHOULD BE CALLED BEFORE ANY OTHER OF THE 
   * UID HANDLING FUNCTIONS.
   */
  int sl_policy_get_user(char *username);  /* drop SUID to <username>  */ 
  int sl_policy_get_real(char *username);  /* drop privs to <username> */
  int sl_policy_get_root(void);            /* drop SUID to root        */

  /*
   * If not using one of the above, use this function, 
   * and then call sh_unset_suid().
   * This function saves the uid's.
   * It calls abort() on error.
   */
  int sl_save_uids(void);

  /*
   * This function returns the saved euid.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_get_euid(uid_t * ret);
  uid_t sl_ret_euid();

  /*
   * This function returns the saved egid.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_get_egid(gid_t * ret);

  /*
   * This function returns the saved current ruid.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_get_ruid(uid_t * ret);
  
  /*
   * This function returns the saved current rgid.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_get_rgid(gid_t * ret);

  /*
   * This function returns the saved original ruid.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_get_ruid_orig(uid_t * ret);

  /*
   * This function returns the saved original rgid.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_get_rgid_orig(gid_t * ret);

  /*
   * This function returns true if the program is SUID.
   * It calls abort() if the uid's are not saved already.
   */
  int sl_is_suid(void);

  /*
   * This function sets the effective uid 
   * to the saved effective uid.
   */
  int sl_set_suid (void);

  /*
   * This function sets the effective uid to the real uid.
   */
  int sl_unset_suid (void);

  /* 
   * This function drops SUID privileges irrevocably.
   */
  int sl_drop_privileges(void);

  /* ---------------------------------------------------------------- 
   *
   *    File handling routines
   *
   * ---------------------------------------------------------------- */

  /* Open for writing.
   */
  SL_TICKET  sl_open_write       (char * fname, int priviledge_mode);

  /* Open for reading.
   */
  SL_TICKET  sl_open_read        (char * fname, int priviledge_mode);

  /* Open for reading w/minimum checking.
   */
  SL_TICKET  sl_open_fastread    (char * fname, int priviledge_mode);

  /* Open for read and write.
   */
  SL_TICKET  sl_open_rdwr        (char * fname, int priviledge_mode);

  /* Open for read and write, fail if file exists.
   */
  SL_TICKET sl_open_safe_rdwr    (char * fname, int priv);

  /* Open for write, truncate.
   */
  SL_TICKET  sl_open_write_trunc (char * fname, int priviledge_mode);

  /* Open for read and write, truncate.
   */
  SL_TICKET  sl_open_rdwr_trunc  (char * fname, int priviledge_mode);

  /* Close file.
   */
  int sl_close (SL_TICKET ticket);

  /* Unlink file.
   */
  int sl_unlink (SL_TICKET ticket);

  /* Rewind file.
   */
  int sl_rewind (SL_TICKET ticket);

  /* Seek file.
   */
  int sl_seek (SL_TICKET ticket, off_t off_data);
 
  /* Forward file.
   */
  int sl_forward (SL_TICKET ticket);

  /* Sync file.
   */
  int sl_sync (SL_TICKET ticket);

  /* Read file.
   */
  int sl_read (SL_TICKET ticket, void * buf, size_t count);

  /* Write file.
   */
  int sl_write (SL_TICKET ticket, void * msg, long nbytes);

  /* Write file, terminate with newline.
   */
  int sl_write_line (SL_TICKET ticket, void * msg, long nbytes);

  /* Drop all metadata for file descriptors >= fd.
   */
  int sl_dropall(int fd);

  /* Check whether file is trustworthy.
   */
  int sl_trustfile(char * path, long * ok, long * bad);

  /* Check whether file is trustworthy.
   */
  int sl_trustfile_euid(char * filename, long euid);

  /* Add a trusted user.
   */
  int  sl_trust_add_user (long pwid);

  /* Get error string.
   */
  char * sl_error_string(int errorcode);

  /* Get error file.
   */
  char * sl_trust_errfile(void);


#ifdef  __cplusplus
}
#endif

/* Privilege modes for file access.
 */
#define SL_YESPRIV 0x33
#define SL_NOPRIV  0x34

#define MAXFILENAME	2048


/*
 * This macro is TRUE if (x) < 0.
 */
#define SL_ISERROR(x) ((long)(x) < 0)

#if defined(WITH_TPT) 

#define TPT(arg) dlog arg ;

#else

#define TPT(arg)

#endif

extern int myblock_flag;
#define MBLK(arg)                                      \
do {                                                   \
  if (myblock_flag == 0){                              \
    myblock_flag = 1;                                  \
    arg ;                                              \
    myblock_flag = 0;                                  \
  }                                                    \
} while (0);

/*
 * The 'require' macro.
 */
#define SL_REQUIRE(assertion, astext)                  \
do {                                                   \
    if (assertion) ;                                   \
    else {                                             \
        dlog(0, FIL__, __LINE__, SDG_AFAIL,            \
                 FIL__, __LINE__, astext);             \
        _exit(EXIT_FAILURE);                           \
    }                                                  \
} while (0)


/*
 * The enter macro. Prints the trace if TRACE is on.
 */
#if defined(SL_DEBUG)
#define SL_ENTER(s)  sl_stack_push(s);
#else
#define SL_ENTER(s)  ;
#endif

/*
 * The return macro.
 */
#if defined(SL_DEBUG)
#define SL_RETURN(x, s)   \
do {                      \
   sl_stack_pop();        \
   if (1) return(x);      \
} while(0)

#else

#define SL_RETURN(x, s)   \
do {                      \
   if (1) return(x);      \
} while(0)

#endif  /* SL_RETURN macro */

#if defined(SL_DEBUG)
#define SL_RET0(s)      \
do {                    \
      sl_stack_pop();   \
      if (1) return;    \
} while(0)

#else
#define SL_RET0(s)      \
do {                    \
      if (1) return;    \
} while(0)
#endif  /* SL_RETURN macro */

#if defined(SL_DEBUG)
void sl_stack_push(char * c);
void sl_stack_pop();
void sl_stack_print();
#endif




/*
 * The internal return macro. Sets sl_errno to the return value.
 */

#if defined(SL_DEBUG)
#define SL_IRETURN(x, s)                                            \
do {                                                                \
   if((long)(x) < 0) {                                              \
      TPT((0,    FIL__, __LINE__, SDG_ERROR, (long)(x)))     \
      sl_errno=(x);                                                 \
    }                                                               \
   sl_stack_pop();                                                  \
   if (1) return(x);                                                \
} while(0)
#else
#define SL_IRETURN(x, s)             \
do {                                 \
   if ((long)(x) < 0) sl_errno=(x);  \
   if (1) return(x);                 \
} while(0)

#endif  /* SL_IRETURN macro */



/* slib.h */
#endif 




